import os, sys
import time
import ConfigParser

import ctypes
import sdl2
import sdl2.ext
import sdl2.sdlttf


def justify(s, l):
    """ returns a string of l length """
    if len(s) < l:
        return s + " "*(l - len(s))
    else:
        return s[:l]


### Classes

class Colors():
    # COLORS
    white = sdl2.SDL_Color(255, 255, 255, 255)
    black = sdl2.SDL_Color(0, 0, 0, 255)


class Config():
    width = 640
    height = 480
    font_size = 16
    back_color = Colors.black
    fore_color = Colors.white
    show_line_numbers = True
    font = "Hack-Regular.ttf"


class State():
    """ The highest level state obejct """
    def __init__(self):
        self.bufs = []
        self.curbuf = None
        # windows size in pixels
        self.width  = 0
        self.height = 0
        # in pixels
        self.cwidth  = 0
        self.cheight = 0
        # windows size in chars
        self.rows = 0
        self.columns = 0
        # cursor pos in chars
        self.x = 0
        self.y = 0
        
        # status message
        self.statusmsg = ""
        self.status_refresh = False
        self.status_timer = None

        # menu navigation
        self.curmenu     = None
        self.curmenuitem = None

        # line numbers
        self.leftmargin = 3

    def addBuf(self, buf):
        self.bufs.append(buf)
        if self.curbuf is None:
            self.switchBuffer(0)

    def switchBuffer(self, idx):
        self.curbuf = idx
        self.updateStatusBar()

    def updateStatusBar(self, s=None):
        if s is None:
            fn = justify(os.path.split(self.bufs[self.curbuf].filename)[1], 16)
            if self.bufs[self.curbuf].dirty:
                dirty = "!"
            else:
                dirty = " "
            msg1 = "[E]:%s%s" % (fn, dirty)
            msg2 = "(%d, %d) / (%d, %d)" % (self.x, self.y, self.columns, self.rows)
            #TODO: handle very small windows
            spacer = " "*(self.columns - len(msg1) - len(msg2))
            self.statusmsg = msg1 + spacer + msg2
        else:
            self.statusmsg = justify(s, self.columns)
        self.status_refresh = True
        self.status_timer = time.time()

    def drawRow(self):
        pass

    def drawRows(self):
        pass


class Buffer():
    """ This class contains the rows for a particular buffer """
    def __init__(self, filename=""):
        self.filename = filename
        self.dirty = False
        self.rows = []
        if filename != "":
            self.loadFile(filename)

    def addRow(self, s, pos=-1):
        if pos == -1:
            pos = len(self.rows)
        self.rows.insert(pos, Row(s, pos))

    def loadFile(self, fn):
        self.filename = fn
        self.rows = []
        for line in open(fn).readlines():
            self.addRow(line.rstrip())

    def saveFile(self, fn):
        outfile = open(fn, "w")
        outfile.writelines([r.raw for r in self.rows])
        outfile.close()

    def len(self):
        return len(self.rows)
            

class Row():
    """ A single row of text """
    def __init__(self, s, y):
        self.raw = ""
        self.render = ""
        self.y = y
        self.refresh = False
        self.surface = None
        self.update(s)

    def update(self, s):
        self.raw = s
        self.render = s
        self.refresh = True

    def len(self):
        return len(self.render)


def loadConfig():
    cp = ConfigParser.SafeConfigParser()
    cp.read(["ca.ini", os.path.expanduser("~/.config/ca.ini")])

def drawText(font, s):
    """ draw a surface at position x,y """
    return sdl2.sdlttf.TTF_RenderText_Shaded(font, s, 
            Config.fore_color, Config.back_color)

def main(args):
    state = State()

    #RESOURCES = sdl2.ext.Resources(__file__, ".")

    # INITIALIZE
    sdl2.ext.init()
    sdl2.sdlttf.TTF_Init()

    window = sdl2.ext.Window("Crabapple Editor", size=(Config.width, Config.height))
    window.show()

    # LOAD FONTS
    font = sdl2.sdlttf.TTF_OpenFont(Config.font, Config.font_size)
    sdl2.sdlttf.TTF_SetFontHinting(font, sdl2.sdlttf.TTF_HINTING_MONO)
    # NORMAL, LIGHT, MONO, NONE

    # compute some sizes
    s = "Hello World"
    w = ctypes.c_int()
    h = ctypes.c_int()
    sdl2.sdlttf.TTF_SizeText(font, s, ctypes.byref(w), ctypes.byref(h))
    # window size in pixels
    state.width  = Config.width
    state.height = Config.height
    # character size in pixels
    state.cwidth = w.value / len(s)
    state.cheight= h.value
    # total screen size in rows/columns
    state.rows   = state.height / state.cheight
    state.columns= state.width / state.cwidth
    
    # viewport size
    state.viewrows    = state.rows - 1
    state.viewcolumns = state.columns
    # offset in pixels (to account for margins)
    state.viewoffsetx = 0
    state.viewoffsety = 0
    # current scroll position
    state.scrollrow    = 0
    state.scrollcolumn = 0

    windowsurface = sdl2.SDL_GetWindowSurface(window.window)

    if len(args) > 0 and os.path.exists(args[0]):
        print(args[0])
        newbuf = Buffer(args[0]) 
        state.addBuf(newbuf)
        state.updateStatusBar()
    else:
        state.updateStatusBar("Welcome to Crabapple Editor")

    status_y = (state.rows - 1) * state.cheight
    statussurface = None

    running = True
    while running:
        events = sdl2.ext.get_events()
        for event in events:
            if event.type == sdl2.SDL_QUIT:
                running = False
                break
            elif event.type == sdl2.SDL_KEYDOWN:
                if event.key.keysym.sym == sdl2.SDLK_q and \
                    sdl2.SDL_GetModState() & sdl2.KMOD_CTRL:
                    running = False
                    break
                elif event.key.keysym.sym == sdl2.SDLK_UP:
                    if state.y > 0:
                        state.y -= 1
                elif event.key.keysym.sym == sdl2.SDLK_DOWN:
                    if state.y < state.bufs[state.curbuf].len() - 1:
                        state.y += 1

        sdl2.SDL_Delay(10)

        if state.status_refresh:
            print("status bar drawing")
            if statussurface is not None:
                sdl2.SDL_FreeSurface(statussurface)
            statussurface = drawText(font, state.statusmsg)
            sdl2.SDL_BlitSurface(statussurface, None, windowsurface, 
                    sdl2.SDL_Rect(0, status_y, state.width, state.cheight))
            state.status_refresh = False

        if state.curbuf is not None:
            buf = state.bufs[state.curbuf]
            for ri, row in enumerate(buf.rows[state.y: state.y+state.viewrows]):
                if row.refresh:
                    if row.surface is not None:
                        sdl2.SDL_FreeSurface(row.surface)
                    row.surface = drawText(font, justify(row.render, state.viewcolumns))
                y = ri * state.cheight + state.viewoffsety
                x = state.viewoffsetx
                sdl2.SDL_BlitSurface(row.surface, None, windowsurface,
                    sdl2.SDL_Rect(x, y, state.width, state.cheight))

        sdl2.SDL_UpdateWindowSurface(window.window)
        window.refresh()


    sdl2.SDL_FreeSurface(surface)
    sdl2.sdlttf.TTF_CloseFont(font)
    sdl2.sdlttf.TTF_Quit()
    sdl2.ext.quit()

if __name__ == "__main__":
    main(sys.argv[1:])
