from oasis.core.keyprocessor import KeyProcessor, ActionCode
from oasis.core.modemachine import Modes
from oasis.core.bindings import Bindings
from oasis.core.actions import Action, Actions
from oasis.core.keyevent import keybuffer_from_strings as kb, string_to_key




def test_keyprocessor_nav():
    mode = Modes.navigate
    bindings = Bindings()

    # some actions are specific to the widget
    Actions.append("Boom")
    Actions.append("yank_line", register="0")
    Actions.append("record_macro", register=None)
    Actions.append("yank_doc_end")
    Actions.append("find_char", char=None)

    # some actions are required
    bindings.add({
        kb("esc"):      Actions.abortkey,
        kb("y", "y"):   Actions.yank_line,
        kb("q"):        Actions.record_macro,
        kb("y", "G"):   Actions.yank_doc_end,
        kb("f"):        Actions.find_char,
        kb("a", "a", "a"): Actions.Boom,
        }, 
        mode)
    for k in bindings.bindings[mode]["global"]:
        print(str(k), str(bindings.bindings[mode]["global"][k]))

    kp = KeyProcessor(mode, bindings)

    # test escape
    key = string_to_key("esc")
    result, action = kp.process_key("global", key)
    assert(action == Actions.abortkey)
    key = string_to_key("a")
    result, action = kp.process_key("global", key)
    key = string_to_key("esc")
    result, action = kp.process_key("global", key)
    assert(action == Actions.abortkey)

    # test a valid command
    key = string_to_key("a")
    result, action = kp.process_key("global", key)
    assert(result == ActionCode.no_action)
    # possible results: no_action, passthru, action, invalid
    result, action = kp.process_key("global", key)
    assert(result == ActionCode.no_action)
    result, action = kp.process_key("global", key)
    assert(result == ActionCode.action)
    assert(action == Actions.Boom)

    # test escape
    key = string_to_key("esc")
    result, action = kp.process_key("global", key)
    assert(action == Actions.abortkey)

    # test an invalid command
    kp.reset()
    keyb = string_to_key("b")
    result, action = kp.process_key("global", key)
    assert(result == ActionCode.no_action)
    result, action = kp.process_key("global", keyb)
    assert(result == ActionCode.invalid)
    assert(len(kp.buf) == 0)

    # test a command with length 2
    kp.reset()
    result, action = kp.process_key("global", string_to_key("y"))
    assert(result == ActionCode.no_action)
    result, action = kp.process_key("global", string_to_key("y"))
    assert(result == ActionCode.action)
    assert(action.name == "yank_line")

    # test a command with a numeric prefix
    """
    kp.reset()
    result, action = kp.process_key(string_to_key("4"))
    assert(result == ActionCode.no_action)
    result, action = kp.process_key(string_to_key("y"))
    assert(result == ActionCode.no_action)
    result, action = kp.process_key(string_to_key("y"))
    assert(result == ActionCode.action)
    assert(action.name == "yank_line")
    assert(action.count == "4")
    """
    """
    # test with a register prefix
    kp.reset()
    prefix = string_to_key('"')
    result, action = kp.process_key(prefix)
    assert(result == ActionCode.no_action)
    result, action = kp.process_key(string_to_key("1"))
    assert(result == ActionCode.no_action)
    result, action = kp.process_key(string_to_key("y"))
    assert(result == ActionCode.no_action)
    result, action = kp.process_key(string_to_key("y"))
    assert(result == ActionCode.action)
    assert(action.name == "yank_line")
    assert(action.register == "1")
    """
    """
    # test with a register suffix
    kp.reset()
    result, action = kp.process_key(string_to_key("q"))
    assert(result == ActionCode.no_action)
    result, action = kp.process_key(string_to_key("a"))
    assert(result == ActionCode.action)
    assert(action.name == "record_macro")
    assert(action.register == "a")
    """
    """
    # test a compound command (with a motion suffix)
    kp.reset()
    result, action = kp.process_key(string_to_key("y"))
    assert(result == ActionCode.no_action)
    result, action = kp.process_key(string_to_key("G"))
    assert(result == ActionCode.action)
    assert(action.name == "yank_doc_end")
    """
    """
    # test find char command
    kp.reset()
    result, action = kp.process_key(string_to_key("f"))
    assert(result == ActionCode.no_action)
    result, action = kp.process_key(string_to_key("G"))
    assert(result == ActionCode.action)
    assert(action.name == "find_char")
    assert(action.char == "G")
    """

def test_keyprocessor_cmd():
    mode = Modes.command
    bindings = Bindings()

    Actions.add("run command")

    # some actions are required
    bindings.add({
        kb("esc"):      Actions.abortkey,
        kb("return"):   Actions["run command"]
        }, 
        mode)

    kp = KeyProcessor(mode, bindings)

    result, action = kp.process_key("global", string_to_key("esc"))
    assert(action == Actions.abortkey)
    # verify it does not buffer passthru actions
    result, action = kp.process_key("global", string_to_key("a"))
    result, action = kp.process_key("global", string_to_key("esc"))
    assert(action == Actions.abortkey)
