""" wm.gui - Window Manager widget for oasis """
import os
import logging
from oasis.core.guihelpers import (
    keyPressHandlerForApp,
    keyPressHandlerForContainer,
    keyPressHandlerForEndPoint,
    validateGui
)

from PySide2.QtWidgets import (
    QMainWindow,
    QWidget,
    QVBoxLayout,
    QHBoxLayout,
    QStatusBar,
    QLineEdit,
    QLabel,
    QSizePolicy,
    QCompleter,
    QFileSystemModel,
)

from PySide2.QtCore import (
        Qt,
)

from oasis.wm.tree import Tree, Node

STYLESHEET = \
'''
QWidget {{
    background-color: #{background:06x};
    color: #{foreground:06x};
    font-family: "{font}";
    font-size: {font-size}px;
    border: 1px solid {border};
    padding: {padding};
    margin: {margins};
}}
'''


class WidgetNode(Node):
    normStyle = '''
    QWidget {
        border-width: 1px;
        border-style: solid;
        border-color: black;
    }
    '''
    selStyle = '''
    QWidget {
        border-width: 3px;
        border-style: solid;
        border-color: black;
    }
    '''
    MAPLO = {Node.H: QHBoxLayout, Node.V: QVBoxLayout}
    def __init__(self, parent=None, kind=Node.H):
        Node.__init__(self, parent, kind)
        self.lo = WidgetNode.MAPLO[kind]()
        self.lo.setContentsMargins(0, 0, 0, 0)
        self.lo.setSpacing(0)

    def insert(self, i, child):
        Node.insert(self, i, child)
        if isinstance(child, WidgetNode):
            # child is another WidgetNode
            # Was getting an error like "layout already has parent"
            # then changed a check for hasattr(child, "lo") to isinstance
            # not sure what was going on
            self.lo.insertLayout(i, child.lo)
        else:
            # child is itself a QWidget
            self.lo.insertWidget(i, child)

    def remove(self, i):
        child = Node.remove(self, i)
        self.lo.removeWidget(child)
        return child

    def unselect(self):
        w = self.children[self.index]
        while not hasattr(w, "setStyleSheet"):
            w = w.cur_child()
        w.setStyleSheet(WidgetNode.normStyle)

    def select(self, i):
        Node.select(self, i)
        w = self.children[i]
        while not hasattr(w, "setStyleSheet"):
            w = w.cur_child()
        w.setStyleSheet(WidgetNode.selStyle)


class Cli(QLineEdit):
    def __init__(self):
        QLineEdit.__init__(self)
        self.prompt = ":"
        self.fileCompleter = None
        self.commandCompleter = None

        self.fileCompleter = QCompleter()
        self.fileCompleter.setCompletionMode(QCompleter.InlineCompletion)
        self.fileModel = QFileSystemModel()
        self.fileModel.setRootPath(os.getcwd())
        self.fileCompleter.setModel(self.fileModel)

    def setupCompleters(self, commandList, path):
        self.commandCompleter = QCompleter(list(commandList))
        self.commandCompleter.setCompletionMode(QCompleter.InlineCompletion)

    def requestFileComplete(self, prefix):
        logging.debug("wm.gui.Cli: using fileCompleter")
        return self._requestComplete(self.fileCompleter, prefix)

    def requestCommandComplete(self, prefix):
        logging.debug("wm.gui.Cli: using commandCompleter")
        return self._requestComplete(self.commandCompleter, prefix)

    def _requestComplete(self, completer, prefix):
        completer.setCompletionPrefix(prefix)
        completer.complete()
        # fires a signal that we ignore
        logging.debug("wm.gui.Cli: Completion requested: %s, %s" % (prefix, completer.currentCompletion()))
        # faster way to check for a single completion than completionCount()
        if not completer.setCurrentRow(1) and completer.setCurrentRow(0):
            return completer.currentCompletion()
        else:
            # TODO: popup suggestions?
            return None

    def keyPressEvent(self, event):
        if not self.parent().keyPressEvent(event):
            cursor_x = self.cursorPosition()
            key = event.key()
            # TODO: maybe this should be caught in the controller?
            if key == Qt.Key_Home:
                self.setCursorPosition(len(self.prompt))
                return False
            if cursor_x == len(self.prompt):
                if key in [Qt.Key_Backspace, Qt.Key_Left]:
                    return False
            return QLineEdit.keyPressEvent(self, event)
        return True

    def reset(self):
        self.setText(self.prompt)
        self.setReadOnly(False)
        self.setFocus()

    def unfocus(self):
        self.clear()
        self.clearFocus()
        self.setReadOnly(True)

    def text(self):
        return QLineEdit.text(self)[len(self.prompt):]

    def setCommand(self, cmd):
        self.setText(self.prompt + cmd)


class Status(QWidget):
    def __init__(self, parent):
        QWidget.__init__(self, parent)
        self.setSizePolicy(QSizePolicy.Expanding, QSizePolicy.Fixed)
        self.setStyleSheet("border: 1px solid red; padding: 0; margin: 0;")
        self.setContentsMargins(0,0,0,0)
        self.lo = QVBoxLayout(self)
        # these two lines were sufficient for removing all spacing b/t
        # widgets in the status bar
        self.lo.setContentsMargins(0, 0, 0, 0)
        self.lo.setSpacing(0)
        self.message = QLabel()
        self.message.setSizePolicy(QSizePolicy.Expanding, QSizePolicy.Fixed)
        self.status = QLabel()
        self.status.setSizePolicy(QSizePolicy.Expanding, QSizePolicy.Fixed)
        self.cli = Cli()
        self.lo.addWidget(self.message)
        self.lo.addWidget(self.cli)
        self.lo.addWidget(self.status)

    def keyPressEvent(self, event):
        return keyPressHandlerForContainer(self, event)


class WrappedTool(QWidget):
    def __init__(self, w):
        QWidget.__init__(self)
        lo = QVBoxLayout(self)
        titleBar = QLabel()
        lo.addWidget(titleBar)
        lo.addWidget(w)
        self.setText = titleBar.setText
        self.unwrapped = w


class WmMainWidget(QWidget):
    def __init__(self):
        QWidget.__init__(self)
        self.nextsplit = Node.H
        self.tree = Tree(WidgetNode)
        self.lo = QHBoxLayout(self)
        self.lo.setContentsMargins(0, 0, 0, 0)
        self.lo.setSpacing(0)

    def keyPressEvent(self, event):
        return keyPressHandlerForContainer(self, event)

    def addWidget(self, w):
        if self.tree.cur_node is not None:
            self.tree.cur_node.unselect()
        if self.tree.cur_node is None:
            first = True
        else:
            first = False
        self.tree.insert(w, self.nextsplit)
        if first:
            if self.lo.count() > 0:
                child = self.lo.takeAt(0)
                while self.lo.takeAt(0):
                    child = self.lo.takeAt(0)
                    del child
            self.lo.addLayout(self.tree.cur_node.lo)

    def selectedWidget(self):
        child = self.tree.selected().cur_child()
        if hasattr(child, "unwrapped"):
            return child.unwrapped
        else:
            return child

    def changeSplitV(self):
        self.nextsplit = Node.V

    def changeSplitH(self):
        self.nextsplit = Node.H

    def moveSelectedLeft(self):
        self.tree.move_selected(Tree.DIR_LEFT)

    def moveSelectedRight(self):
        self.tree.move_selected(Tree.DIR_RIGHT)

    def moveSelectedUp(self):
        self.tree.move_selected(Tree.DIR_UP)

    def moveSelectedDown(self):
        self.tree.move_selected(Tree.DIR_DOWN)

    def changeSelectedLeft(self):
        self.tree.change_selected(Tree.DIR_LEFT)

    def changeSelectedRight(self):
        self.tree.change_selected(Tree.DIR_RIGHT)

    def changeSelectedUp(self):
        self.tree.change_selected(Tree.DIR_UP)

    def changeSelectedDown(self):
        self.tree.change_selected(Tree.DIR_DOWN)

    def removeSelected(self):
        w = self.tree.remove()
        if w is not None:
            w.hide()


class WmGui(QWidget):
    def __init__(self):
        QWidget.__init__(self)
        self.setWindowTitle("Oasis")
        self.wm = WmMainWidget()
        self.status = Status(self)
        self.requestFileComplete = self.status.cli.requestFileComplete
        self.requestCommandComplete = self.status.cli.requestCommandComplete
        self.setCmd = self.status.cli.setCommand
        self.lo = QVBoxLayout(self)
        self.lo.setContentsMargins(0,0,0,0)
        self.lo.setSpacing(0)
        self.lo.addWidget(self.wm)
        self.lo.addWidget(self.status)

    def style(self, yamlstyle):
        ss = STYLESHEET.format(**yamlstyle)
        logging.debug(ss)
        self.setStyleSheet(ss)

    def show(self):
        self.showMaximized()

    def showEvent(self, event):
        validateGui(self)
        QMainWindow.showEvent(self, event)

    def addWidget(self, w):
        self.wm.addWidget(w)

    def wrapWidget(self, w):
        """ Takes a QWidget (derivative) and wraps it in a QVBoxLayout with a 
        pseudo title bar.
        """
        return WrappedTool(w)

    def keyPressEvent(self, event):
        return keyPressHandlerForApp(self, event)

    def getCmd(self):
        cmd = self.status.cli.text()
        return cmd

    def resetCmd(self):
        self.status.cli.reset()

    def enableCmd(self):
        self.status.cli.reset()

    def disableCmd(self):
        self.status.cli.unfocus()

    def setStatus(self, msg):
        self.status.status.setText(msg)

