""" 
    wm.ctrl - As i3-like Window Manager layout engine for oasis 

"""
import os
import logging
from oasis.core.abstracttool import AbstractTool
from oasis.core.actions import Actions
from oasis.core.commandparser import Command


class WmTool(AbstractTool):
    def __init__(self, model):
        AbstractTool.__init__(self, model)
        self.action_handlers = {
            Actions.add("quit all"): self.quitAll,
            Actions.add("save quit tool"): self.removeSelected,
            Actions.add("force quit tool"): self.removeSelected,
            Actions.add("run command"): self.runCommand,
            Actions.add("command mode"): self.enterCommandMode,
            Actions.add("navigate mode"): self.enterNavigateMode,
            Actions.add("insert mode"): self.enterInsertMode,
            Actions.add("vertical layout"): self.changeSplitV,
            Actions.add("horizontal layout"): self.changeSplitH,
            Actions.add("focus left"): self.changeSelectedLeft,
            Actions.add("focus down"): self.changeSelectedDown,
            Actions.add("focus up"): self.changeSelectedUp,
            Actions.add("focus right"): self.changeSelectedRight,
            Actions.add("move widget left"): self.moveSelectedLeft,
            Actions.add("move widget down"): self.moveSelectedDown,
            Actions.add("move widget up"): self.moveSelectedUp,
            Actions.add("move widget right"): self.moveSelectedRight,
            Actions.add("backward history"): self.backwardHistory,
            Actions.add("foreward history"): self.forewardHistory,
            Actions.add("suggest completion"): self.suggest,
        }
        self.commands = [
            Command("quit", self.quitAll),
        ]
        self.commandParser = None

        self.hasfocus = None    # the current widget with focus
        self.commandHistory = []
        self.commandIndex = None

    ## API
    def setCommandParser(self, parser):
        self.commandParser = parser

    def addTool(self, a):
        wrapper = self.gui.wrapWidget(a.gui)
        self.gui.addWidget(wrapper)
        self.hasfocus = a
        a.setTitle = wrapper.setText

    def show(self):
        self.gui.style(self.model.config.style)
        self.gui.show()

    ## Action Handlers
    def quitAll(self, action):
        self.gui.close()

    def runCommand(self, action):
        cmd = self.gui.getCmd()
        self.gui.resetCmd()
        if self.commandParser is None:
            raise Exception("WM: Command has no parser set")
        self.commandHistory.append(cmd)
        self.commandIndex = len(self.commandHistory)
        self.commandParser(self.hasfocus, cmd)
        self.enterNavigateMode(None)

    def enterCommandMode(self, action):
        self.commandIndex = len(self.commandHistory)
        self.gui.enableCmd()
        self.gui.setStatus("Mode: command")

    def enterInsertMode(self, action):
        self.gui.setStatus("Mode: insert")

    def enterNavigateMode(self, action):
        self.gui.disableCmd()
        self.gui.setStatus("Mode: navigate")

    def changeSplitV(self, action):
        self.gui.wm.changeSplitV()

    def changeSplitH(self, action):
        self.gui.wm.changeSplitH()

    def moveSelectedLeft(self, action):
        self.gui.wm.moveSelectedLeft()

    def moveSelectedRight(self, action):
        self.gui.wm.moveSelectedRight()

    def moveSelectedUp(self, action):
        self.gui.wm.moveSelectedUp()

    def moveSelectedDown(self, action):
        self.gui.wm.moveSelectedDown()

    def changeSelectedLeft(self, action):
        self.gui.wm.changeSelectedLeft()
        w = self.gui.wm.selectedWidget()
        self.hasfocus = w.ctrl

    def changeSelectedRight(self, action):
        self.gui.wm.changeSelectedRight()
        w = self.gui.wm.selectedWidget()
        self.hasfocus = w.ctrl

    def changeSelectedUp(self, action):
        self.gui.wm.changeSelectedUp()
        w = self.gui.wm.selectedWidget()
        self.hasfocus = w.ctrl

    def changeSelectedDown(self, action):
        self.gui.wm.changeSelectedDown()
        w = self.gui.wm.selectedWidget()
        self.hasfocus = w.ctrl

    def removeSelected(self, action):
        #TODO: Handle a failed save
        # if action was a save/quit, verify that w.ctrl.saved == True
        self.gui.wm.removeSelected()
        w = self.gui.wm.selectedWidget()
        self.hasfocus = w.ctrl

    def backwardHistory(self, action):
        if len(self.commandHistory) == 0:
            return
        self.commandIndex -= 1
        if self.commandIndex < 0:
            self.commandIndex = len(self.commandHistory) - 1
        self.gui.setCmd(self.commandHistory[self.commandIndex])

    def forewardHistory(self, action):
        if len(self.commandHistory) == 0:
            return
        self.commandIndex += 1
        if self.commandIndex >= len(self.commandHistory):
            self.commandIndex = 0
        self.gui.setCmd(self.commandHistory[self.commandIndex])

    def suggest(self, action):
        """ suggest a command or filename """
        # you have to update the completers when this is run because the valid
        # commands may have changed if tools have been opened/closed
        self.gui.status.cli.setupCompleters(self.commandParser.commandList(),
                "/home/brian")
        cmd = self.gui.getCmd()
        tokens = cmd.split()
        if len(tokens) == 1:
            suggestion = self.gui.requestCommandComplete(cmd)
            if suggestion is not None:
                self.gui.setCmd(suggestion)
        elif len(tokens) == 2:
            cmd, args = self.commandParser.getCommand(self.hasfocus, cmd)
            if cmd and "filename" in cmd.args:
                fn = tokens[1]
                if not os.path.exists(os.path.split(fn)[0]):
                    fn = os.path.join(os.path.abspath(os.getcwd()), fn)
                logging.debug("wm.ctrl: attempting to complete: %s" % fn)
                suggestion = self.gui.requestFileComplete(fn)
                if suggestion is not None:
                    self.gui.setCmd(" ".join((tokens[0], suggestion)))
