""" Files/ctrl.py """
import os, stat
from datetime import datetime
from libs.classutil import requiresNotNone, requiresCallable
from libs import shell
from libs.humanize import humanize
from oasis.files.const import *
from oasis.core.abstracttool import AbstractTool

class FilesTool(AbstractTool):
    def __init__(self, model):
        AbstractTool.__init__(self, model)
        self.gui = None
        self.cwd = os.getcwd()
        self.longView = False

    def attachGui(self, gui):
        self.gui = gui
        api = [
            "setStatus",
            "populateTable",
            "currentFilename",
            "startEditing",
            "stopEditing",
            "moveCursorUp",
            "moveCursorDown",
            "showLongView",
            "showShortView",
            "startVisual",
            "stopVisual",
            "rangeFilenames",
        ]
        for func in api:
            requiresCallable(self.gui, func)

    def registerActions(self, manager):
        """ registers handlers from this controller with the associated actions """
        pass

    def openFile(self, fn):
        pass

    def setCwd(self, cwd):
        requiresNotNone(self, "gui")
        self.gui.setStatus(cwd)
        self.cwd = cwd
        self.listing()

    def listing(self):
        dirs = [".."]
        files = []
        for f in sorted(os.listdir(self.cwd)):
            if os.path.isdir(f):
                dirs.append(f)
            else:
                files.append(f)
        results = []
        for f in dirs+files:
            row = list([""]*NUMCOLS)
            s = os.lstat(os.path.join(self.cwd, f))
            row[ColumnIndexes.MODE] = stat.filemode(s.st_mode)
            row[ColumnIndexes.USER] = shell.getUserFromUid(s.st_uid)
            row[ColumnIndexes.SIZE] = humanize(s.st_size)
            row[ColumnIndexes.DATE] = datetime.fromtimestamp(s.st_ctime).strftime("%Y%m%d %H:%M")
            row[ColumnIndexes.GROUP] = str(s.st_gid)
            row[ColumnIndexes.FILENAME] = f
        self.currentListing = results
        self.gui.populateTable(results)

    ### Action Handlers
    def handleEnter(self):
        requiresNotNone(self, "gui")
        fn = self.gui.currentFilename()
        if os.path.isdir(fn):
            self.setCwd(fn)
        else:
            self.openFile(fn)

    def cursorUp(self):
        self.gui.moveCursorUp()

    def cursorDown(self):
        self.gui.moveCursorDown()

    def visualUp(self):
        pass

    def visualDown(self):
        pass

    def yank(self):
        """ Check for range selected else just current fn """
        pass

    def delete(self):
        pass

    def put(self):
        pass

    def rename(self):
        pass

    def touch(self):
        # create a file
        pass

    def mkdir(self):
        # create a folder
        pass

    def toggleLong(self):
        self.longView = not self.longView
        if self.longView:
            self.gui.showLongView()
        else:
            self.gui.showShortView()
        
    def enterVisual(self):
        pass

    def exitVisual(self):
        pass

    def gotoTop(self):
        pass

    def gotoEnd(self):
        pass

    def gotoLine(self):
        pass

    def gotoFilename(self):
        pass
