""" Editor - Pretty self explanatory """
import math

from PySide2.QtWidgets import QWidget, QHBoxLayout, QTextEdit
from PySide2.QtGui import (
        QSyntaxHighlighter,
        QTextBlockUserData,
        QTextCharFormat,
        QTextCursor,
        QBrush,
        QColor,
        QFont,
        QPainter,
        QPen,
    )
from PySide2.QtCore import Qt

from pygments.token import Token
from pygments.lexers import PythonLexer

from oasis.core.guihelpers import (
    keyPressHandlerForApp,
    keyPressHandlerForContainer,
    keyPressHandlerForEndPoint,
    validateGui
)

def makeFormat(foreground, background=None,
        italic=False, bold=False, underline=False):
    """ create a QTextCharFormat based on some common attributes """
    tcFormat = QTextCharFormat()
    tcFormat.setForeground(QBrush(QColor(foreground)))
    if background:
        tcFormat.setBackground(QBrush(QColor(background)))
    if italic:
        tcFormat.setFontItalic(True)
    if bold:
        tcFormat.setFontWeight(100)
    return tcFormat


class UserData(QTextBlockUserData):
    """ Stores the highlighter state of the current textblock """
    def __init__(self, stack):
        QTextBlockUserData.__init__(self)
        self.stack = stack


class Highlighter(QSyntaxHighlighter):
    """ Pygments based highlighter class """
    def __init__(self, doc):
        QSyntaxHighlighter.__init__(self, doc)
        self.lex = PythonLexer()
        # WARNING: a typo on Token.XX not throw an error
        self.styles = {
                Token.Comment: makeFormat(0x00AA00, italic=True),
                Token.Literal.String.Doc: makeFormat(0x00AA00, italic=True),
                Token.Keyword: makeFormat(0x6600AA),
                Token.Name: makeFormat(0x00),
                Token.Name.Function: makeFormat(0xAA5500),
                Token.Name.Class: makeFormat(0xAA5500, bold=True),
                Token.Literal.String: makeFormat(0x1100AA),
                Token.Literal: makeFormat(0xAA0000),
                Token.Operator: makeFormat(0xAA0000),

        }

    def styleMatcher(self, token):
        match = None
        for tokenType in self.styles:
            if token in tokenType and \
                (not match or len(tokenType) > len(match)):
                match = tokenType
        return match


    def highlightBlock(self, text):
        block = self.currentBlock()
        userdata = block.previous().userData()
        if not userdata or not userdata.stack:
            stack = ('root',)
        else:
            stack = userdata.stack
        for r in self.lex.get_tokens_unprocessed(text, stack):
            i, tt, v, stack = r
            match = self.styleMatcher(tt)
            if match:
                self.setFormat(i, len(v), self.styles[match])
        self.setCurrentBlockUserData(UserData(stack))


class ModalTextEdit(QTextEdit):
    """ adds some features to QTextEdit to support a modal interface """

    def __init__(self):
        QTextEdit.__init__(self)
        self.fakeCursorRect = None
        self.fillCursor = False
        self.cursorPositionChanged.connect(self._updateCursorRect)

    def paintEvent(self, event):
        QTextEdit.paintEvent(self, event)
        if not self.fakeCursorRect:
            return
        painter = QPainter()
        painter.begin(self.viewport())
        painter.setCompositionMode(QPainter.CompositionMode_Difference)
        if self.fillCursor:
            painter.setBrush(QBrush(QColor(0xFFFFFF)))
        else:
            pen = QPen(QColor(0xFFFFFF))
            pen.setWidth(1)
            painter.setPen(pen)
        painter.drawRect(self.fakeCursorRect)
        painter.end()

    def resizeEvent(self, event):
        # you can't do this from init
        QTextEdit.resizeEvent(self, event)
        self._updateCursorRect()

    def keyPressEvent(self, event):
        keyPressHandlerForEndPoint(QTextEdit, self, event)

    def _updateCursorRect(self):
        if self.isReadOnly():
            self.cursorPositionChanged.disconnect()
            start = self.cursorRect()
            tc = self.textCursor()
            startpos = tc.position()
            emptyBlock = False
            if tc.atBlockEnd():
                if tc.atBlockStart():
                    # kids don't try this at home
                    emptyBlock = True
                    tc.insertText(" ")
                else:
                    tc.movePosition(QTextCursor.Left)
            else:
                tc.movePosition(QTextCursor.Right)
            self.setTextCursor(tc)
            self.fakeCursorRect = self.cursorRect().united(start)
            if emptyBlock:
                tc.deletePreviousChar()
            else:
                tc.setPosition(startpos)
            self.setTextCursor(tc)
            self.update() # without this the widget doesn't redraw everything
            # but it does redraw the cursor?!?
            self.cursorPositionChanged.connect(self._updateCursorRect)
        else:
            self.fakeCursorRect = None

    def setText(self, text):
        QTextEdit.setText(self, text)
        self._updateCursorRect()

    def setInsert(self, pos):
        self.setReadOnly(False)
        self.setFocus()
        self.fakeCursorRect = None
        if pos == "above line":
            tc = self.textCursor()
            tc.movePosition(QTextCursor.StartOfLine)
            tc.insertText("\n")
            tc.movePosition(QTextCursor.PreviousCharacter)
            self.setTextCursor(tc)
        elif pos == "below line":
            tc = self.textCursor()
            tc.movePosition(QTextCursor.EndOfLine)
            tc.insertText("\n")
            self.setTextCursor(tc)

    def setNavigateFocus(self):
        self.setReadOnly(True)
        self.clearFocus()
        self._updateCursorRect()
        self.fillCursor = True

    def setNavigateNoFocus(self):
        self.setReadOnly(True)
        self.clearFocus()
        self._updateCursorRect()
        self.fillCursor = False

    def insertString(self, s):
        tc = self.textCursor()
        tc.insertText(s)

    def backspaceCharClass(self, charClass):
        tc = self.textCursor()
        pos = min(tc.anchor(), tc.position())
        tc.setPosition(pos)
        if tc.positionInBlock() == 0:
            return
        tc.movePosition(QTextCursor.PreviousCharacter, QTextCursor.KeepAnchor)
        char = tc.selectedText()
        if char in charClass:
            tc.removeSelectedText()


class EditGui(QWidget):
    """ The GUI for the editor """
    def __init__(self):
        QWidget.__init__(self)
        self.lo = QHBoxLayout(self)
        self.lo.setContentsMargins(0, 0, 0, 0)
        self.lo.setSpacing(0)
        self.main = ModalTextEdit()
        self.nums = QTextEdit()
        self.nums.setFixedWidth(40)
        self.nums.setVerticalScrollBarPolicy(Qt.ScrollBarAlwaysOff)
        self.nums.setHorizontalScrollBarPolicy(Qt.ScrollBarAlwaysOff)
        self.nums.setReadOnly(True)
        self.nums.setFocusPolicy(Qt.NoFocus)

        # link the main text scrollbar to the line number scrollbar
        self.main.verticalScrollBar().valueChanged.connect(
                self._updateNums)

        self.main.document().blockCountChanged.connect(self._updateNums)

        #TODO: replace with user config font 
        self.main.document().setDefaultFont(QFont("Monospace", 14))
        self.lo.addWidget(self.nums)
        self.lo.addWidget(self.main)
        self.hl = Highlighter(self.main.document())

        self.main.setFocusPolicy(Qt.NoFocus)
        self.disableEditing()

        # handle pass thru methods
        self.insertString = self.main.insertString
        self.backspaceCharClass = self.main.backspaceCharClass

    def showEvent(self, event):
        validateGui(self)
        QWidget.showEvent(self, event)

    def keyPressEvent(self, event):
        return keyPressHandlerForApp(self, event)

    def _updateNums(self, *args):
        # update the line numbers when the main document, block count changes
        # OR when it scrolls b/c QTextDocument does lazy layout
        doc = self.main.document()
        count = doc.blockCount()
        width = int(math.log(count, 10)) + 1
        self.nums.setFixedWidth(14 * width + 8)
        linenums = []
        for n in range(count):
            linenums.append(str(n).rjust(width))
            tb = doc.findBlockByNumber(n)
            lines = tb.layout().lineCount()
            if lines > 1:
                linenums.extend([" "*width for f in range(lines-1)])
        self.nums.setPlainText("\n".join(linenums))
        self.nums.verticalScrollBar().setValue(self.main.verticalScrollBar().value())

    ### API
    def setText(self, text):
        self.main.setText(text)

    def getText(self):
        return self.main.toPlainText()

    def enableEditing(self, pos="cursor"):
        self.main.setInsert(pos)

    def disableEditing(self):
        self.main.setNavigateFocus()

    def moveCursorUp(self, num):
        if num == -1:
            self.main.moveCursor(QTextCursor.Start)
            return
        for f in range(num):
            self.main.moveCursor(QTextCursor.Up)

    def moveCursorDown(self, num):
        if num == -1:
            self.main.moveCursor(QTextCursor.End)
            return
        for f in range(num):
            self.main.moveCursor(QTextCursor.Down)

    def moveCursorLeft(self, num):
        if num == -1:
            # should this be start of block?
            self.main.moveCursor(QTextCursor.StartOfLine)
            return
        for f in range(num):
            self.main.moveCursor(QTextCursor.Left)

    def moveCursorRight(self, num):
        #TODO: in Nav mode you should NOT be able to move the cursor past the last char of the line
        # causes the "double" move bug
        if num == -1:
            self.main.moveCursor(QTextCursor.EndOfLine)
            return
        for f in range(num):
            self.main.moveCursor(QTextCursor.Right)

