import os
import logging

from oasis.core.abstracttool import AbstractTool
from oasis.core.commandparser import Command
from oasis.core.actions import Actions


class EditTool(AbstractTool):
    def __init__(self, model):
        AbstractTool.__init__(self, model)
        self.action_handlers = {
            Actions.add("insert mode"): self.beginInsert,
            Actions.add("insert above line"): self.beginInsert,
            Actions.add("insert below line"): self.beginInsert,
            Actions.add("navigate mode"): self.endInsert,
            Actions.add("cursor up"): self.cursorUp,
            Actions.add("cursor down"): self.cursorDown,
            Actions.add("cursor left"): self.cursorLeft,
            Actions.add("cursor right"): self.cursorRight,
            Actions.add("end of line"): self.cursorEndLine,
            Actions.add("start of line"): self.cursorStartLine,
            Actions.add("cursor page up"): self.cursorPageUp,
            Actions.add("cursor page down"): self.cursorPageDown,
            Actions.add("top"): self.cursorTop,
            Actions.add("bottom"): self.cursorBottom,
            Actions.add("insert tab"): self.insertTab,
            Actions.add("antitab"): self.backTab,
        }

        self.commands = [
            Command("open", self.openFile, ["filename"]),
            Command("save", self.saveFile, ["filename"])
        ]
        # TODO: pull page/up down amount from config
        self.linesInPage = 16

        self.filename = None

    def onLoad(self):
        self.setTitle("edit")

    def openFile(self, args):
        fn = args["filename"]
        if not os.path.exists(fn):
            logging.error("%s does not exist" % fn)
            return
        self.gui.setText(open(fn).read())
        self.filename = fn
        self.setTitle("edit: {}".format(fn))

    def saveFile(self, args):
        text = self.gui.getText()
        if not "filename" in args:
            if not self.filename:
                # TODO: message user
                return
            fn = self.filename
        else:
            fn = args["filename"]
        with open(fn, "w") as fo:
            fo.write(text)

    def beginInsert(self, action):
        if "above line" in action.name:
            self.gui.enableEditing("above line")
        elif "below line" in action.name:
            self.gui.enableEditing("below line")
        else:
            self.gui.enableEditing()

    def endInsert(self, action):
        self.gui.disableEditing()

    def cursorUp(self, action):
        self.gui.moveCursorUp(1)

    def cursorDown(self, action):
        self.gui.moveCursorDown(1)

    def cursorLeft(self, action):
        self.gui.moveCursorLeft(1)

    def cursorRight(self, action):
        self.gui.moveCursorRight(1)

    def cursorEndLine(self, action):
        self.gui.moveCursorRight(-1)

    def cursorStartLine(self, action):
        self.gui.moveCursorLeft(-1)

    def cursorPageUp(self, action):
        self.gui.moveCursorUp(self.linesInPage)

    def cursorPageDown(self, action):
        self.gui.moveCursorDown(self.linesInPage)

    def cursorTop(self, action):
        self.gui.moveCursorUp(-1)

    def cursorBottom(self, action):
        self.gui.moveCursorDown(-1)

    def insertTab(self, action):
        # TODO: pull from config
        # TODO: handle visual mode
        self.gui.insertString(" " * 4)

    def backTab(self, action):
        # TODO: pull from config
        # TODO: handle visual mode
        for f in range(4):
            self.gui.backspaceCharClass(" \t")

