import logging

from oasis.core.keyconst import KeyConst
from oasis.core.keyevent import KeyEvent
from oasis.core.keyprocessor import KeyProcessor, ActionCode
from oasis.core.actions import Actions


class Mode():
    def __init__(self, name,
            passthru=False,
            accept_int_prefix=False,
            accept_register_prefix=False):
        self.name = name
        # passthru=True means that keys that are not bound to actions
        # are passed back to the widget that captured them ie. a TextEdit
        self.passthru = passthru
        self.accept_int_prefix = accept_int_prefix
        self.accept_register_prefix = accept_register_prefix

    def __str__(self):
        return self.name


class Modes():
    command     = Mode("command", passthru=True)
    insert      = Mode("insert", passthru=True)
    navigate    = Mode("navigate", accept_int_prefix=True, accept_register_prefix=True)
    resize      = Mode("resize")
    visual      = Mode("visual")


class ModeMachine():
    """ ModeMachine is the main controller object for oasis 
    """
    def __init__(self, bindings):
        self.kp = KeyProcessor(Modes.navigate, bindings)
        self.action_handlers = {"global":{}}
        self.wm = None
        Actions.add("navigate mode")
        Actions.add("command mode")
        Actions.add("run command")
        Actions.add("insert mode")
        Actions.add("insert above line")
        Actions.add("insert below line")

        for action, func in [
                (Actions["navigate mode"], self.enterNavigateMode),
                (Actions["command mode"], self.enterCommandMode),
                (Actions["run command"], self.enterNavigateMode),
                (Actions["insert mode"], self.enterInsertMode),
                (Actions["insert above line"], self.enterInsertMode),
                (Actions["insert below line"], self.enterInsertMode),
                ]:
            self.register_handler(action, "global", func)

    def setWm(self, wm):
        self.wm = wm

    ### Action Handlers
    def enterNavigateMode(self, action):
        self.kp.change_mode(Modes.navigate)

    def enterCommandMode(self, action):
        self.kp.change_mode(Modes.command)

    def enterInsertMode(self, action):
        self.kp.change_mode(Modes.insert)

    ### The keyevent handler
    def handle_keyevent(self, tool, keyevent):
        # translate a KeyEvent into an Action (if it is one)
        # maybe I should not be relying on which QWidget generated the event
        # but rather which tool has the focus
        if self.wm:
            tool = self.wm.hasfocus
        if tool != "global":
            result, action = self.kp.process_key(tool.identity(), keyevent)
        else:
            result, action = self.kp.process_key(tool, keyevent)
        logging.debug("(%s: %s) from %s" % (result, action, tool))
        if result == ActionCode.no_action:
            return True    # should be true?
        elif result == ActionCode.passthru:
            return False
        elif result == ActionCode.invalid:
            # report to user?
            return True
        elif result == ActionCode.action:
            # emit!
            self.dispatch_action(action, tool)
            return True
        else:
            raise Exception("Unsupported ActionCode received from KeyProcessor")

    ### Action Dispatcher
    def dispatch_action(self, action, tool):
        # call registered handlers for the given action
        # check if the WM wants this action
        # don't dispatch actions to self.wm twice
        objs = ["global", self.wm]
        if tool != self.wm:
            objs.append(tool)
        for obj in objs:
            if not obj in self.action_handlers:
                continue
            if not action in self.action_handlers[obj]:
                continue
            for handler in self.action_handlers[obj][action]:
                logging.debug("ModeMachine dispatches: %s to %s" % (action, handler))
                handler(action)

    ### Register a function that handles an Action
    def register_handler(self, action, tool, handler):
        if not tool in self.action_handlers:
            self.action_handlers[tool] = {}
        if not action in self.action_handlers[tool]:
            self.action_handlers[tool][action] = []
        self.action_handlers[tool][action].append(handler)
        



