import os
import sys
import logging
import argparse

from oasis.core.modemachine import ModeMachine
from oasis.core.commandparser import CommandParser, Command
from oasis.core.config import Config

from oasis.core.toolclasses import ToolClassRegistry
from oasis.core.actions import Actions

from oasis.core.qtapp import App

mainparser = argparse.ArgumentParser(
    description="oasis - a haven of consistent interfaces"
)
mainparser.add_argument("--debug", action="store_true", help="set logging level to debug")
mainparser.add_argument("files", nargs="*", help="files to open on startup")

class Main():
    def __init__(self):
        self.tools = []

    def start(self, filepath):
        args = mainparser.parse_args()
        if args.debug:
            logging.getLogger().setLevel(logging.DEBUG)
        self.filepath = filepath
        #TODO: process args
        # Load config here
        self.config = Config(self.filepath)
        self.modemachine = ModeMachine(self.config.bindings)
        self.commandparser = CommandParser()

        self.app = App()

        self.wm = self.spawnTool("wm")
        self.wm.setCommandParser(self.commandparser)

        # load any files passed on the command line (if we can)
        if args.files:
            for filename in args.files:
                name, ext = os.path.splitext(filename)
                ext = ext[1:]  # strip off leading dot
                if ext in self.config.file_associations:
                    tool = self.spawnAndAdd(self.config.file_associations[ext])
                    tool.openFile({"filename": filename})
                else:
                    logging.warning("filetype: {}, of file {} is not associated with a tool.".format(ext, filename))
        else:
            self.spawnAndAdd("inspecter")

        self.wm.show()

        self.modemachine.setWm(self.wm)
        self.commandparser.setWm(self.wm)

        # setup some global actions
        for action, func in [
                (Actions["spawn inspector"], 
                    lambda action: self.spawnAndAdd("inspecter")),
                (Actions["spawn editor"], 
                    lambda action: self.spawnAndAdd("edit")),
                (Actions["spawn files"],
                    lambda action: self.spawnAndAdd("files")),
                (Actions["spawn web"],
                    lambda action: self.spawnAndAdd("web")),
                (Actions["spawn terminal"],
                    lambda action: self.spawnAndAdd("terminal")),
                (Actions["spawn table"],
                    lambda action: self.spawnAndAdd("table")),
                ]:
            self.modemachine.register_handler(action, "global", func)

        for cmd in [
                Command("spawn", self.handleSpawnCmd, ["tool"]),
                ]:
            self.commandparser.addCommand("global", cmd)
                    

        # set the initial mode
        self.modemachine.dispatch_action(Actions["navigate mode"], None)

        self.app.start()

    def spawnTool(self, module):
        if module == "wm" and hasattr(self, "wm"):
            raise ValueError("You cannot spawn more than one WM")
        widget = ToolClassRegistry[module].spawn(
                self, \
                self.keyEventHandler, \
                self.modemachine.register_handler, \
                self.commandparser.addCommand
            )
        return widget

    def spawnAndAdd(self, module):
        tool = self.spawnTool(module)
        self.tools.append(tool)
        self.wm.addTool(tool)
        if hasattr(tool, "onLoad"):
            tool.onLoad()
        return tool

    def handleSpawnCmd(self, args):
        toolname = args["tool"]
        if not toolname in ToolClassRegistry:
            logging.error("Cannot spawn tool: %s" % toolname)
        else:
            self.spawnAndAdd(toolname)

    def keyEventHandler(self, event, w):
        logging.debug("Main.keyEventHandler by: %s with: %s" % (w, event))
        return self.modemachine.handle_keyevent(w, event)

