import logging
from enum import Enum

from oasis.core.keyevent import KeyBuffer
from oasis.core.actions import Actions


class ActionCode(Enum):
    # return codes from Processors
    no_action = 0
    passthru = 1
    action = 2
    invalid = 3


# These are the processors for keyevents, given the mode
class KeyProcessorStates(Enum):
    pass
"""
    Keyprocessor extensions:
        WAIT -> a number starts buffering a count [0-9]
             -> " starts waiting for a register [0-9a-zA-Z]
        After a partial match and if action requires
             -> register name [0-9a-zA-Z]
             -> motion command
             -> single character
    wait_prefix = 2
    wait_action = 3
    wait_suffix = 4
    state_machine
            KeyProcessorStates.wait_prefix: self.wait_prefix_state,
            KeyProcessorStates.wait_suffix: self.wait_suffix_state,
            KeyProcessorStates.wait_action: self.wait_action_state,
    def init_state(self, key):
        # check for state transistions
        kb = KeyBuffer()
        kb.append(key)
        action = self.bindings.match(kb, self.mode)
        if key.isnum() and self.mode.accept_int_prefix:
            self.state = KeyProcessorStates.wait_prefix
            self.charclass = "isnum"
            self.num = -1
            return self.wait_prefix_state(key)
        elif action == Actions.prefix_register \
                and self.mode.accept_register_prefix:
            self.state = KeyProcessorStates.wait_prefix
            self.charclass = "isalphanum"
            self.num = 1
            return (ActionCode.no_action, None)
        else:
            return self.wait_action_state(key)

    def wait_action_state(self, key):
        self.buf.append(key)
        partial = False
        for k, action in self.bindings.get_all(self.mode):
            result = k.match(self.buf)
            if result == KeyBuffer.exact_match:
                #TODO: go to wait_suffix if action requires it
                self.prepare_action(action)
                self.reset()
                return (ActionCode.action, action)
            elif result == KeyBuffer.partial_match:
                # at least one was a partial match
                partial = True
        if self.mode.passthru:
            return (ActionCode.passthru, None)
        if partial:
            return (ActionCode.no_action, None)
        else:
            kb = self.buf
            self.reset()
            return (ActionCode.invalid, kb)

    def wait_prefix_state(self, key):
        test = getattr(key, self.charclass)
        if test():
            self.prefix_buf += str(key)
            if self.num > 0 and len(self.prefix_buf) >= self.num:
                self.state = KeyProcessorStates.wait_action
            return (ActionCode.no_action, None)
        elif self.num == -1:
            return self.wait_action_state(key)
        else:
            kb = self.buf
            self.reset()
            return (ActionCode.invalid, kb)

    def wait_suffix_state(self, key):
        test = getattr(key, self.charclass)
        if test():
            self.suffix_buf += str(key)
            if self.num > 0 and len(self.suffix_buf) >= self.num:
                self.prepare_action(action)
                self.reset()
                return (ActionCode.action, action)
            return (ActionCode.no_action, None)
        elif self.num == -1:
            return self.wait_action_state(key)
        else:
            kb = self.buf
            self.reset()
            return (ActionCode.invalid, kb)

    def prepare_action(self, action):
        if action.accept_prefix_int and len(self.prefix_buf) > 0:
            try:
                count = int(self.prefix_buf)
            except:
                pass
            action.count = count
"""

class KeyProcessor():
    """ 
        KeyProcessor matches the current keybuffer (a sequence of keyevents)
        against the bindings object. A partial match means wait for more keyevents
        An exact match results in a action being returned.
    """
    def __init__(self, initmode, bindings):
        self.bindings = bindings
        self.mode = initmode
        self.reset()

    def change_mode(self, mode):
        self.reset()
        self.mode = mode

    def process_key(self, toolId, key):
        kb = KeyBuffer(key)
        action = self.bindings.match(kb, self.mode)
        if action == Actions.abortkey:
            self.reset()
            return (ActionCode.no_action, Actions.abortkey)

        partial = False
        self.buf.append(key)
        actions = list(self.bindings.get_all(self.mode))
        if toolId != "wm":
                actions += list(self.bindings.get_all(self.mode, "wm"))
        if toolId != "global":
                actions += list(self.bindings.get_all(self.mode, toolId))
        #logging.debug("Mode: %s has bindings %s" % (self.mode, actions))
        for k, action in actions:
            result = k.match(self.buf)
            if result == KeyBuffer.exact_match:
                self.reset()
                return (ActionCode.action, action)
            elif result == KeyBuffer.partial_match:
                # at least one was a partial match
                partial = True
        if self.mode.passthru:
            self.buf.pop()
            return (ActionCode.passthru, None)
        if partial:
            return (ActionCode.no_action, None)
        else:
            kb = self.buf
            self.reset()
            return (ActionCode.invalid, kb)

    def reset(self):
        self.buf = KeyBuffer()




