from oasis.core.keyconst import KeyConst, Mods, KeyAliases, ImpliedMods

def string_to_key(s):
    # assume string is properly validated
    assert(type(s) is str)
    mods = 0
    if "+" in s and len(s) > 1:
        fields = s.split("+")
        if len(fields) > 1:
            s = fields.pop()
            for f in fields:
                mods += Mods[f]
    if s in KeyAliases:
        s = KeyAliases[s]
    if "A" <= s <= "Z":
        mods += Mods["shift"]
        s = s.lower()
    elif s in ImpliedMods:
        mods += ImpliedMods[s]
    if s in KeyConst:
        return KeyEvent(KeyConst[s], mods)
    else:
        raise Exception("Invalid key string: %s" % s)

class KeyEvent():
    def __init__(self, key=None, mods=None, ev=None):
        """ a wrapped key event """
        self.key = None
        self.modifiers = 0
        self.ev = None
        if key is not None:
            self.key = key
        if mods is not None:
            self.modifiers = mods
        if ev is not None:
            self.ev = ev

    def isprintable(self):
        return KeyConst["a"] <= self.key < KeyConst["space"]

    def isnum(self):
        return KeyConst["1"] <= self.key <= KeyConst["0"]

    def isalpha(self):
        return KeyConst["a"] <= self.key <= KeyConst["z"]

    def isalphanum(self):
        return self.isnum() or self.isalpha()

    def inset(self, chkset):
        # checks if this key is in chkset (a list)
        for key in chkset:
            if self == key:
                return True
        return False

    def __hash__(self):
        return self.key+self.modifiers

    def __eq__(self, other):
        return self.key == other.key and self.modifiers == other.modifiers

    def __str__(self):
        result = ""
        if self.modifiers > 0:
            for k in Mods.names():
                if self.modifiers & Mods[k]:
                    result += k + "+"
        return result+KeyConst[self.key]


class KeyBuffer():
    """ A KeyBuffer is a sequence of KeyEvents
        Matchable and appendable, they facilitate the vim-like
        behavior of oasis
    """
    no_match = 0
    partial_match = 1
    exact_match = 2
    def __init__(self, keyevent=None):
        self.buf = []
        if keyevent is not None:
            self.append(keyevent)

    def clear(self):
        self.buf = []

    def match(self, other):
        # detects a match between this keysequence and the other
        # If the other match is longer, they do not match
        if len(other) > len(self):
            return KeyBuffer.no_match
        result = None
        for this_key, other_key in zip(self.buf, other.buf):
            if this_key == other_key:
                result = KeyBuffer.exact_match
            else:
                # if any key doesn't match that's a fail
                return KeyBuffer.no_match

        if len(self) > len(other):
            # if the other buffer is shorter but matches so far, that's a partial
            return KeyBuffer.partial_match
        else:
            return KeyBuffer.exact_match

    def append(self, key):
        self.buf.append(key)

    def pop(self, i=-1):
        return self.buf.pop(i)

    def __len__(self):
        return len(self.buf)

    def __getitem__(self, i):
        return self.buf[i]

    def __hash__(self):
        h = 0
        for i, k in enumerate(self.buf):
            h += hash(k)*(0x1000**i)
        return h

    def __eq__(self, other):
        return self.__hash__() == other.__hash__()

    def __str__(self):
        return " ".join([str(s) for s in self.buf])


def keybuffer_from_strings(*args):
    """ Takes a list of strings and returns a KeyBuffer """
    kb = KeyBuffer()
    for s in args:
        k = string_to_key(s)
        kb.append(k)
    return kb

