import os
import yaml
import logging

from oasis.core.keyevent import keybuffer_from_strings
from oasis.core.actions import Actions
from oasis.core.modemachine import Modes
from oasis.core.bindings import Bindings
from oasis.core.toolclasses import ToolClassRegistry


class Config():
    def __init__(self, filepath):
        """ first check the XDG_ define locations
            default to filepath/data/default.yaml
        """
        self.bindings = Bindings()
        # TODO: XDG_ locations

        # load default config files
        path = os.path.join(filepath, "data")

        for fn, method in [("config.yaml", self._populate_config),
                ("bindings.yaml", self._populate_bindings),
                ("style.yaml", self._populate_style),
                ("filetypes.yaml", self._associate_filetypes)]:
            self._load_yaml_cfg(
                os.path.join(path, fn),
                method)

    def _load_yaml_cfg(self, filename, method):
        with open(filename, "r") as stream:
            try:
                theyaml = yaml.load(stream)
            except Exception as e:
                raise Exception("%s failed to load: %s" % (filename, e))
            else:
                method(theyaml)

    def _populate_config(self, theyaml):
        self.currentStyleKey = theyaml["style"]

    def _populate_bindings(self, theyaml):
        for toolname, bindingsByMode  in theyaml.items():
            if toolname != "global":
                tool = ToolClassRegistry[toolname].identity()
            else:
                tool = "global"
            for modename, bindings in bindingsByMode.items():
                mode = getattr(Modes, modename)
                for k, v in bindings.items():
                    kb = keybuffer_from_strings(*k.split())
                    if not v in Actions:
                        Actions.add(v)
                    self.bindings.add({kb: Actions[v]}, mode, tool)

    def _populate_style(self, theyaml):
        # TODO: validate
        self.styles = theyaml
        self.style = self.styles[self.currentStyleKey]

    def _associate_filetypes(self, theyaml):
        self.file_associations = {}
        for key, exts in theyaml.items():
            if not key in ToolClassRegistry:
                logging.error("Invalid tool found in filetypes.yaml: {}".format(key))
                continue
            if not ToolClassRegistry[key].ctrlHasMethod("openFile"):
                logging.error("In order to associate filetypes with tool: {}, it must support the openFile command".format(key))
                continue
            for ext in exts:
                self.file_associations[ext] = key


