import logging
import shlex


class Command():
    """ A command """
    def __init__(self, name, handlerFunc, args=[]):
        """ 
            name = command name (what you type to run it)
            handlerFunc = The function to call to handle the command
        """
        self.name = name
        self.handler = handlerFunc
        assert(callable(handlerFunc))
        self.args = args


class CommandParser():
    """ 
        CommandParser - parses commands that are typed in the WM CLI
    """
    def __init__(self):
        self.commands = {}
        self.wm = None

    def setWm(self, wm):
        self.wm = wm

    def addCommand(self, tool, cmd):
        """
            tool: a AbstractTool instance for the tool that will handle this command
                may also be "global", which will be tried if the tool with focus
                did not register a handler for the command
            cmd: a Command instance
        """
        if not cmd.name in self.commands:
            self.commands[cmd.name] = {}
        self.commands[cmd.name][tool] = cmd

    def commandList(self):
        return self.commands.keys()

    def getCommand(self, focustool, cmdstring):
        """
            Will attempt to parse the command string and return the command and
            it's arguments.
        """
        args = shlex.split(cmdstring)
        if len(args) == 0:
            return None, None
        cmdarg = args[0]
        if not cmdarg in self.commands:
            logging.debug("Invalid command: %s" % cmdarg)
            return None, None
        objs = [focustool, self.wm, "global"]
        for obj in objs:
            if not obj in self.commands[cmdarg]:
                continue
            cmd = self.commands[cmdarg][obj]
            argdata = dict(zip(cmd.args, args[1:]))
            return cmd, argdata
        return None, None

    def __call__(self, focustool, cmdstring):
        """
            will attempt to parse the command string: cmdstring
            focustool is the Ctrl instance for the tool holding focus
            if focustool does not handle the passed in command try "global"
            TODO: return something useful for invalid commands
        """
        logging.debug("CommandParser parsing: %s" % cmdstring)
        cmd, argdata = self.getCommand(focustool, cmdstring)
        if cmd:
            cmd.handler(argdata)
            return True
        logging.debug("Unhandled command: %s" % cmdstring)
        return

