import os
from source.fuzzysearch import FuzzySearch, fileWalker


"""
    Mistakes I've made repeatedly
        1. Signals must be on a QObject (QThread is not a subclass of QObject)
        2. You must call QObject.__init__
        The error you get in both cases is connect (or emit) not an attribute of Signal, which makes no sense.

    I found that my generator was completing in less than BUFFER_TIME but the commandReady signal (triggered by FETCH_PAGE) was being ignored because it came before the event loop in the child thread was started. So I added a final call to processCommands after the generator but before the event loop.

    check_params_cb seems to always yield an error of "Signal emitted within timeout..." even if it really wasn't emitted. Troubleshooting seems to require removing the callback.
"""


def test_fetchPage(qtbot, tmpdir):
    def checkResults(results):
        r = list(results)
        assert len(r) == 10
        return True

    # create FuzzySearcher and prime commandQ
    fz = FuzzySearch(fileWalker("/home"))
    # start the process and wait for pageReady
    with qtbot.waitSignal(
        fz.pageReady, timeout=200, check_params_cb=checkResults
    ) as waiter:
        fz.fetchPage(0, 10)
        fz.start()
    fz.stopThread()
    qtbot.waitUntil(fz.child.isFinished, timeout=200)


def test_changeGenerator(qtbot, tmpdir):
    def checkResults(results):
        r = list(results)
        assert (None, "haiku.txt") in r
        return True

    # create FuzzySearcher
    fz = FuzzySearch(fileWalker("."))
    # start the process and wait for pageReady
    with qtbot.waitSignal(
        fz.pageReady, timeout=200, check_params_cb=checkResults
    ) as waiter:
        # tests a case where results will be returned before the buffer timer expires
        fz.changeGenerator(fileWalker("data"))
        fz.fetchPage(0, 20)
        fz.start()
    fz.stopThread()
    qtbot.waitUntil(fz.child.isFinished, timeout=200)


def test_changeSearch(qtbot, tmpdir):
    def checkResults(results):
        r = list(results)
        assert "haiku.txt" in r[0][1]
        assert len(r) == 1
        return True

    # create FuzzySearcher
    fz = FuzzySearch(fileWalker("."))
    # start the process and wait for pageReady
    with qtbot.waitSignal(
        fz.pageReady, timeout=1000, check_params_cb=checkResults
    ) as waiter:
        fz.changeSearch("data.haik.+txt$")
        fz.fetchPage(0, 20)
        fz.start()
    fz.stopThread()
    qtbot.waitUntil(fz.child.isFinished, timeout=200)


def test_stopThreadViaTimer(qtbot, tmpdir):
    # create FuzzySearcher tests a case where buffer timer will expire
    fz = FuzzySearch(fileWalker("/home"))
    fz.start()
    fz.stopThread()
    qtbot.waitUntil(fz.child.isFinished, timeout=200)
    assert fz.child.isFinished()


def test_stopThreadInEventLoop(qtbot):
    # stop event loop
    fz = FuzzySearch(fileWalker("data"))
    fz.start()
    fz.stopThread()
    qtbot.waitUntil(fz.child.isFinished, timeout=200)
    assert fz.child.isFinished()


def test_addEntry(qtbot, tmpdir):
    TESTENTRY = "aaaaa"

    def checkResults(results):
        r = list(results)
        assert (None, TESTENTRY) in r
        return True

    fz = FuzzySearch(fileWalker("data"))
    fz.start()
    fz.addEntry(TESTENTRY)
    with qtbot.waitSignal(
        fz.pageReady, timeout=200, check_params_cb=checkResults
    ) as waiter:
        fz.fetchPage(0, 20)
    # test the first entry is selected
    assert fz.selected == TESTENTRY
    fz.stopThread()
    qtbot.waitUntil(fz.child.isFinished, timeout=200)


def test_removeEntry(qtbot, tmpdir):
    TESTENTRY = "haiku.txt"

    def checkResults(results):
        r = list(results)
        assert not (None, TESTENTRY) in r
        return True

    fz = FuzzySearch(fileWalker("data"))
    fz.start()
    fz.removeEntry(TESTENTRY)
    with qtbot.waitSignal(
        fz.pageReady, timeout=200, check_params_cb=checkResults
    ) as waiter:
        fz.fetchPage(0, 20)
    fz.stopThread()
    qtbot.waitUntil(fz.child.isFinished, timeout=200)


def test_nextEntry(qtbot, tmpdir):
    TESTENTRY1 = "aaaaaa"
    TESTENTRY2 = "aaaaab"

    def checkResults(results):
        r = list(results)
        assert not (None, TESTENTRY) in r
        return True
    
    filenames = os.listdir("data")

    fz = FuzzySearch(fileWalker("data"))
    fz.start()
    fz.addEntry(TESTENTRY1)
    fz.addEntry(TESTENTRY2)
    # You have to wait for the first pageReady, so there is something selected
    # before you try moving the selection
    with qtbot.waitSignal(fz.pageReady, timeout=200) as waiter:
        fz.fetchPage(0, 20)
    fz.selectNext()
    assert fz.selected == TESTENTRY2
    assert fz.numFound == len(filenames) + 2
    fz.selectPrevious()
    assert fz.selected == TESTENTRY1
    fz.stopThread()
    qtbot.waitUntil(fz.child.isFinished, timeout=200)
