import os
import subprocess


""" Abstracts the chosen Version Control System (VCS).
    NOTES:
        hg uses -R to specify repo
        git uses -C
"""


COMMANDMAP = {
    "hg": {
        "opt": ["-R"],
        "init": ["init"],
        "add": ["add"],
        "ls": ["files"],
        "stat": ["stat"],
        "diff": ["diff"],
        "commit": ["commit", "-m"],
        "ident": ["identify"],
    },
    "git": {
        "opt": ["-C"],
        "init": ["init"],
        "add": ["add"],
        "ls": ["ls-files"],
        "stat": ["status", "-s"],
        "diff": ["diff"],
        "commit": ["commit", "-a", "-m"],
        "ident": ["describe", "--tags", "--always"],
    },
}
IGONREFILENAME = {"hg": ".hgignore", "git": ".gitignore"}


def runCommand(command, cwd=None):
    result = subprocess.run(command, capture_output=True, cwd=cwd)
    if result.returncode == 0:
        return result.stdout.decode()
    else:
        raise Exception(
            f"subprocess command: {' ',join(command)} returned code {result.returncode}: {result.stderr}"
        )


def getAvailableVCS():
    avail = []
    for vcs in COMMANDMAP.keys():
        try:
            runCommand([vcs, "--version"])
        except:
            continue
        else:
            avail.append(vcs)
    return avail


def init(vcs, path):
    runCommand([vcs, *COMMANDMAP[vcs]["init"], path])


def add(vcs, path, filename):
    # filename is absolute
    runCommand([vcs, *COMMANDMAP[vcs]["add"], filename], cwd=path)


def listFiles(vcs, path):
    return [
        os.path.abspath(os.path.join(path, f))
        for f in runCommand([vcs, *COMMANDMAP[vcs]["ls"]], cwd=path).splitlines()
    ]


def getModified(vcs, path):
    TRIM = {"git": 3, "hg": 2}
    report = runCommand([vcs, *COMMANDMAP[vcs]["stat"]], cwd=path).splitlines()
    return [
        os.path.abspath(os.path.join(path, line[TRIM[vcs] :]))
        for line in report
        if "M" in line[: TRIM[vcs]]
    ]


def getStatus(vcs, path):
    TRIM = {"git": 3, "hg": 2}
    report = runCommand([vcs, *COMMANDMAP[vcs]["stat"]], cwd=path).splitlines()
    result = {}
    for line in report:
        result[os.path.abspath(os.path.join(path, line[TRIM[vcs] :]))] = line[
            : TRIM[vcs]
        ]
    return result


def commitAll(vcs, path, message):
    # nice of subprocess to quote arguments for us
    runCommand([vcs, *COMMANDMAP[vcs]["commit"], message], cwd=path)


def getIgnores(vcs, path):
    filters = [l for l in open(IGNOREFILENAME).readlines() if not l.startswith("#")]
    if vcs == "hg":
        filters = [f for f in filters if not f.startswith("syntax")]
    return filters
