""" A file listing utility with a familiar name
"""

import os
import glob
import math
import shutil
import argparse
import datetime
from collections import defaultdict

import source.utils.util as util


def compute_column_widths(data, max_width):
    """will compute the number and width of columns to use
    when formatting a list of strings into a table
    given the total line width: max_width.
    takes a list of strings (data) and integer max_width
    returns a tuple of column widths.
    """
    spacing = 2
    lens = [len(f) + spacing for f in data]
    # naively choose number of columns
    N = math.floor(max_width / max(lens))
    if N == 0 or N == 1:
        # special case: if there is a super long filename we won't get any better than 0 or 1
        return (max(lens),)
    finalWs = ()
    maxCols = math.ceil(max_width / min(lens))
    for NN in range(maxCols, 1, -1):
        inCol = math.ceil(len(lens) / NN)
        colWs = []
        for x in range(NN):
            i = x * inCol
            j = (x + 1) * inCol
            if j > len(lens):
                j = len(lens)
            if i >= len(lens):
                break
            colWs.append(max(lens[i:j]))
        total = sum(colWs)
        if total > max_width:
            # nope NN is too many columns, try NN - 1
            continue
        # it worked ok
        finalWs = tuple(colWs)
        break
    return finalWs


def main(paths=None, longflag=False, maxWidth=40):
    sorted_files = defaultdict(list)
    if paths is None or paths == "":
        cwd = os.getcwd()
        sorted_files[""] = os.listdir(cwd)
    else:
        if type(paths) is str:
            paths = [paths]
        expanded_paths = []
        # glob expansion
        for path in paths:
            expanded_paths.extend(glob.glob(path))
        # get all files (and sort by folder)
        for path in expanded_paths:
            if not os.path.exists(path):
                lines.append("cannot access {}: no such file or directory".format(path))
                continue
            if os.path.isdir(path):
                sorted_files[path] = os.listdir(path)
                continue
            folder, filename = os.path.split(path)
            sorted_files[folder].append(filename)

    # sort filenames
    for folder, filenames in sorted_files.items():
        sorted_files[folder] = sorted(filenames)

    # generate results
    for folder, filenames in sorted_files.items():
        if folder != "":
            yield ("")
            yield (folder + ":")

        # short report
        if not longflag:
            # tabulate (this was kind of tricky)
            termW = maxWidth
            colWs = compute_column_widths(filenames, termW)
            N = len(colWs)
            inCol = math.ceil(len(filenames) / N)

            for i in range(inCol):
                line = ""
                for j in range(N):
                    k = j * inCol + i
                    if k >= len(filenames):
                        break
                    fmt = "{:<" + str(colWs[j]) + "}"
                    line += fmt.format(filenames[k])
                yield (line)
            continue

        # long report
        for filename in filenames:
            s = os.lstat(os.path.join(folder, filename))
            size = util.humanize(s.st_size)
            user = util.getuserfromuid(s.st_uid)
            date = datetime.datetime.fromtimestamp(s.st_mtime).strftime("%Y%m%d %H%M%S")
            l = "%s\t%s\t%s\t%s\t%s\t%s" % (
                s.st_mode,
                user,
                s.st_gid,
                size,
                date,
                filename,
            )
            yield (l)
