""" The tool registry tracks all the tools the craftsperson 
    has in the workshop. 
    Can be queried in by tool type, open file, assigned work area, etc
"""
from collections import OrderedDict

from PySide2.QtCore import Qt

from source.basecls import Singleton
from source.dockwidget import ToolDock


class ToolKeys:
    """ Values here must be names in toolfactory.ToolFactories """
    editor = "editor"
    terminal = "terminal"
    fileList = "fileList"
    fileManager = "fileManager"
    pythonHelper = "pythonHelper"
    search =  "search"
    replace = "replace"
    actionLog = "actionLog"


ONE_OF_A_KIND = [
    ToolKeys.terminal,
    ToolKeys.fileList,
    ToolKeys.fileManager,
    ToolKeys.pythonHelper,
    ToolKeys.search,
    ToolKeys.replace,
    ToolKeys.actionLog,
]


PLACES = {
    "left": Qt.LeftDockWidgetArea,
    "right": Qt.RightDockWidgetArea,
    "top": Qt.TopDockWidgetArea,
    "bottom": Qt.BottomDockWidgetArea,
}


class _ToolRegistry:
    def __init__(self):
        self.ctrl = None
        self.gui = None
        self.factory = None
        self.tools = []
        self.places = OrderedDict()  # tool: place
        self.docks = {}  # tool: dock
        self._onTop = None

    def setup(self):
        pass

    def connect_gui(self, ctrl, gui, factory):
        self.ctrl = ctrl
        self.gui = gui
        self.factory = factory

    def allTools(self):
        return tuple(self.tools)

    def inFocus(self):
        for t in self.tools:
            if t.hasFocus():
                return t
        return None

    def byKind(self, kind):
        """returns a tuple of all tools of a kind"""
        return tuple(tool for tool in self.tools if tool.toolKey == kind)

    def byPlace(self, place):
        """returns a tuple of all tools in a place"""
        return tuple(tool for tool in self.tools if self.places[tool] == place)

    def byFilename(self, filename):
        return tuple(tool for tool in self.tools if tool.filename == filename)

    def placeOf(self, tool):
        return self.places[tool]

    def setDock(self, tool, dock):
        self.docks[tool] = dock

    def getDock(self, tool):
        return self.docks[tool]

    def allDocks(self):
        return tuple(dock for dock in self.docks.values())

    def onTop(self):
        return self._onTop

    def updateOnTop(self, tool):
        self._onTop = tool
        gui = self.gui
        gui.lo.setCurrentWidget(tool)
        gui.updateTitleBar(tool.label())
        gui.fileList.changeSelected(tool)

    def nextTool(self):
        current = self.onTop()
        tools = self.byPlace("main")
        i = tools.index(current)
        i += 1
        if i >= len(tools):
            i = 0
        self.updateOnTop(tools[i])

    def backTool(self):
        current = self.onTop()
        tools = self.byPlace("main")
        i = tools.index(current)
        i -= 1
        if i < 0:
            i = len(tools) - 1
        self.updateOnTop(tools[i])

    def addToMain(self, tool):
        self.places[tool] = "main"
        self.gui.lo.addWidget(tool)
        self.updateOnTop(tool)
        tool.setFocus()

    def addToDocks(self, tool, place):
        self.places[tool] = place
        if not tool in self.docks:
            dock = ToolDock(self.ctrl, self.gui, tool)
            self.setDock(tool, dock)
            area = PLACES[place]
            self.gui.addDockWidget(area, dock)
        dock = self.getDock(tool)
        dock.setWidget(tool)
        self.showDock(tool, dock)

    def showDock(self, tool, dock):
        dock.show()
        if self.gui.tabifiedDockWidgets(dock):
            dock.raise_()

    def spawn(self, toolKey, place=None):
        if toolKey in ONE_OF_A_KIND and self.byKind(toolKey):
            tool = self.byKind(toolKey)[0]
        else:
            tool = self.factory.spawn(toolKey, self.ctrl)
            tool.toolKey = toolKey
            self.tools.append(tool)
            if not place:
                place = tool.defaultLocation
            self.places[tool] = place
            if place == "main":
                self.addToMain(tool)
            else:
                self.addToDocks(tool, place)
        return tool

    def removeFromMain(self, tool):
        self.nextTool()
        if self.onTop() == tool:
            # if this is the last, spawn the default
            self.spawn(ToolKeys.editor)
        self.gui.lo.removeWidget(tool)

    def removeFromDocks(self, tool):
        # self.ctrl.gui.removeDockWidget(self.docks[tool])
        self.docks[tool].hide()

    def hide(self, tool):
        if self.places[tool] == "main":
            self.removeFromMain(tool)
        else:
            self.removeFromDocks(tool)

    def remove(self, tool):
        if self.places[tool] == "main":
            self.removeFromMain(tool)
        else:
            self.removeFromDocks(tool)
        self.tools.remove(tool)
        del self.places[tool]
        if tool in self.docks:
            del self.docks[tool]

    def move(self, tool, place):
        if self.places[tool] == "main":
            self.removeFromMain(tool)
            self.addToDocks(tool, place)
        else:
            self.removeFromDocks(tool)
            self.addToMain(tool)

    def toggleTool(self, kind):
        """If tool is not shown, show it.
        If it's shown but not in focus, focus it.
        If it's in focus, hide it.
        NOTE: this only works for one-of-a-kind tools
        """
        tools = self.byKind(kind)
        if not tools:
            self.spawn(kind)  # default place
            return
        tool = tools[0]
        if self.placeOf(tool) == "main":
            if tool.hasFocus():
                pass  # do nothing?
            else:
                self.updateOnTop(tool)
            return
        if tool.hasFocus():
            self.getDock(tool).hide()
        else:
            dock = self.getDock(tool)
            self.showDock(tool, dock)

    def toggleToolBetweenFocusAndDock(self):
        tool = self.inFocus()
        if not tool:
            return
        if self.placeOf(tool) == "main":
            # move to a dock
            place = tool.defaultLocation
            if place == "main":
                place = "right"
            self.move(tool, place)
        else:
            # move to the main
            self.move(tool, "main")


ToolRegistry = Singleton(_ToolRegistry)
