""" subclass of QTextEdit with support for editing Actions """

from PySide2.QtWidgets import QTextEdit, QTextCursor
from PySide2.QtGui import QKeySequence

from source.actions import ActionEnum, ActionDispatcher


class ActionTextEdit(QTextEdit):
    def __init__(self, ctrl):
        QTextEdit.__init__(self)
        self.ctrl = ctrl

        self.tab = "    "

        ## Action handlers
        A = ActionEnum()
        # include all the typical text edit things
        A.delete_left(lambda: self.hasFocus() and self.deleteLeft())
        A.delete_right(lambda: self.hasFocus() and self.deleteRight())
        A.up.addHandler(lambda: self.hasFocus() and self.cursorUp())
        A.down.addHandler(lambda: self.hasFocus() and self.cursorDown())
        A.left.addHandler(lambda: self.hasFocus() and self.cursorLeft())
        A.right.addHandler(lambda: self.hasFocus() and self.cursorRight())
        A.word_left(lambda: self.hasFocus() and self.wordLeft())
        A.word_right(lambda: self.hasFocus() and self.wordRight())
        A.page_up.addHandler(lambda: self.hasFocus() and self.pageUp())
        A.page_down.addHandler(lambda: self.hasFocus() and self.pageDown())
        A.line_home.addHandler(lambda: self.hasFocus() and self.lineHome())
        A.line_end.addHandler(lambda: self.hasFocus() and self.lineEnd())
        A.file_home.addHandler(lambda: self.hasFocus() and self.fileHome())
        A.file_end.addHandler(lambda: self.hasFocus() and self.fileEnd())
        A.copy(lambda: self.hasFocus() and self.copy())
        A.paste(lambda: self.hasFocus() and self.paste())
        a.cut(lambda: self.hasFocus() and self.cut())
        A.copy_line(lambda: self.hasFocus() and self.copyLine())
        A.cut_line(lambda: self.hasFocus() and self.cutLine())
        A.indent(lambda: self.hasFocus() and self.indent())
        A.unindent(lambda: self.hasFocus() and self.unindent())
        A.select_all.addHandler(lambda: self.hasFocus() and self.selectAll())
        # scroll horizontal?
        # zoom?
        # make global?
        A.undo(lambda: self.hasFocus() and self.undo())
        A.redo(lambda: self.hasFocus() and self.redo())

    # A context manager for operations with the textCursor
    class TC:
        def __enter__(self, obj):
            self.obj = obj
            self.tc = obj.textCursor()
            return self.tc

        def __exit__(self):
            self.obj.setTextCursor(self.tc)

    def deleteLeft(self):
        with TC(self) as tc:
            tc.deletePreviousChar()

    def deleteRight(self):
        with TC(self) as tc:
            tc.deleteChar()

    def cursorUp(self):
        with TC(self) as tc:
            tc.movePosition(QTextCursor.Up)

    def cursorDown(self):
        with TC(self) as tc:
            tc.movePosition(QTextCursor.Down)

    def cursorLeft(self):
        with TC(self) as tc:
            tc.movePosition(QTextCursor.Left)

    def cursorRight(self):
        with TC(self) as tc:
            tc.movePosition(QTextCursor.Right)

    def wordLeft(self):
        with TC(self) as tc:
            tc.movePosition(QTextCursor.WordLeft)

    def wordRight(self):
        with TC(self) as tc:
            tc.movePosition(QTextCursor.WordRight)

    def pageUp(self):
        # TODO: compute
        with TC(self) as tc:
            tc.movePosition(QTextCursor.Up, QTextCursor.MoveAnchor, 30)

    def pageDown(self):
        # TODO: compute
        # document().pageSize / Font.lineHeight?
        with TC(self) as tc:
            tc.movePosition(QTextCursor.Down, QTextCursor.MoveAnchor, 30)

    def lineHome(self):
        with TC(self) as tc:
            tc.movePosition(QTextCursor.StartOfBlock)

    def lineEnd(self):
        with TC(self) as tc:
            tc.movePosition(QTextCursor.EndOfBlock)

    def fileHome(self):
        with TC(self) as tc:
            tc.movePosition(QTextCursor.Start)

    def fileEnd(self):
        with TC(self) as tc:
            tc.movePosition(QTextCursor.End)

    """ Already implemented in QTextEdit
    def copy(self):
    def paste(self):
    def cut(self):
    """

    def copyLine(self):
        tc = self.textCursor()
        pos = tc.position()
        tc.movePosition(QTextCursor.StartOfBlock)
        tc.movePosition(QTextCursor.EndOfBlock, QTextCursor.KeepAnchor)
        self.setTextCursor(tc)
        self.copy()
        tc.setPosition(pos)
        self.setTextCursor(tc)

    def cutLine(self):
        tc = self.textCursor()
        pos = tc.position()
        tc.movePosition(QTextCursor.StartOfBlock)
        tc.movePosition(QTextCursor.EndOfBlock, QTextCursor.KeepAnchor)
        self.setTextCursor(tc)
        self.cut()

    def indent(self):
        tc = self.textCursor()
        tc.beginEditBlock()
        if tc.hasSelection():
            # for a selection...
            doc = self.document()
            startBlock = doc.findBlock(tc.selectionStart())
            endBlock = doc.findBlock(tc.selectionEnd())
            nBlocks = endBlock - startBlock
            tc.setPosition(tc.selectionStart())
            for n in range(nBlocks):
                tc.movePosition(QTextCursor.StartOfLine)
                tc.insertText(self.tab)
                tc.movePosition(QTextCursor.Down)
        else:
            self.insertPlainText(self.tab)
        tc.endEditBlock()

    def unindent(self):
        def unindentLine(tc):
            tc.movePosition(QTextCursor.StartOfLine)
            for f in range(len(self.tab)):
                tc.movePosition(QTextCursor.NextCharacter, QTextCursor.KeepAnchor)
            if tc.selectedText() == self.tab:
                tc.deleteChar()

        tc = self.textCursor()
        tc.beginEditBlock()
        if tc.hasSelection():
            doc = self.document()
            startBlock = doc.findBlock(tc.selectionStart())
            endBlock = doc.findBlock(tc.selectionEnd())
            nBlocks = endBlock - startBlock
            tc.setPosition(tc.selectionStart())
        else:
            nBlocks = 1
        for n in range(nBlocks):
            unidentLine(tc)
            tc.movePosition(QTextCursor.Down)
        tc.endEditBlock()

    """ Already implemented in QTextEdit
    def selectAll(self):
    def undo(self):
    def redo(self):
    """

    def keyPressEvent(self, event):
        keyseq = QKeySequence(event.key())
        if ActionDispatcher().dispatchKey(keyseq):
            return True
        # remove this line if/when all editing functions are implemented as actions
        QTextEdit.keyPressEvent(self, event)


