import io
import os
import re
import sys
import shlex
import shutil
import socket
import getpass
from PySide2.QtWidgets import QDockWidget, QTextEdit
from PySide2.QtGui import QTextCursor
from PySide2.QtCore import Qt, Signal, Slot

from source.action import ActionEnum, ActionDispatcher
from source.childthread import GeneratorOutputHandler, SubprocessHandler

# command line utilities
from source.utils import cat, cp, find, grep, ls, md5sum, mv, rm, touch, ztar


class CommandLineInterface(QTextEdit):
    """A generic command line interface that is used in lots of the tools"""

    def __init__(self, caret="$ "):
        QTextEdit.__init__(self)
        # TODO: allow user to configure prompt string
        self.MAX_LINES = 1000
        self.prompt_caret = caret
        self.history = []
        self.historyIndex = 0
        self.showPrompt()

    def saveToHistory(self, cmdString):
        if len(self.history) >= 1000:
            del self.history[-1]
        self.history.insert(0, cmdString)

    def moveToLastLine(self):
        tc = self.textCursor()
        pos = tc.position()
        doc = self.document()
        if doc.findBlock(pos) != doc.lastBlock():
            tc.setPosition(doc.lastBlock().position())
            tc.movePosition(QTextCursor.EndOfLine)
            self.setTextCursor(tc)
        sb = self.verticalScrollBar()
        sb.setValue(sb.maximum())

    def moveToHome(self, select=False):
        tc = self.textCursor()
        pos = tc.block().position() + len(self.ps)
        if select:
            tc.setPosition(pos, QTextCursor.KeepAnchor)
        else:
            tc.setPosition(pos)
        self.setTextCursor(tc)

    def recallCmd(self, forwards=True):
        tc = self.textCursor()
        tc.setPosition(tc.block().position() + len(self.ps))
        tc.movePosition(QTextCursor.EndOfLine, QTextCursor.KeepAnchor)
        tc.insertText(self.history[self.historyIndex])
        self.setTextCursor(tc)
        if forwards:
            self.historyIndex += 1
        else:
            self.historyIndex -= 1
        if self.historyIndex >= len(self.history):
            self.historyIndex = 0
        elif self.historyIndex < 0:
            self.historyIndex = len(self.history) - 1

    def keyPressEvent(self, event):
        # Capture keypresses and modify behavior for a CLI-like interface
        key = event.key()
        # prevent typing off the last line
        self.moveToLastLine()
        if key == Qt.Key_Home:
            self.moveToHome(select=event.modifiers() & Qt.ShiftModifier)
        elif key in (Qt.Key_Backspace, Qt.Key_Left):
            if self.textCursor().positionInBlock() > len(self.ps):
                QTextEdit.keyPressEvent(self, event)
        elif key in (Qt.Key_Up, Qt.Key_Down):
            self.recallCmd(forwards=(key == Qt.Key_Up))
        # TODO: tab completion
        elif key == Qt.Key_Tab:
            pass
        # operator on command
        elif key == Qt.Key_Return:
            self.historyIndex = 0
            self.enterCommand()
        # otherwise handle normally
        else:
            QTextEdit.keyPressEvent(self, event)

    def enterCommand(self):
        tc = self.textCursor()
        tc.movePosition(QTextCursor.StartOfBlock)
        tc.movePosition(QTextCursor.EndOfBlock, QTextCursor.KeepAnchor)
        cmdString = tc.selectedText()[len(self.ps) :]
        tc.movePosition(QTextCursor.EndOfLine)
        self.setTextCursor(tc)
        self.saveToHistory(cmdString)
        self.runCommand(cmdString)

    def showPrompt(self):
        self.moveToLastLine()
        tc = self.textCursor()
        tc.movePosition(QTextCursor.EndOfBlock)
        tc.insertText("\n" + self.prompt())
        self.setTextCursor(tc)

    @Slot(str)
    def appendOutput(self, content):
        lines = content.splitlines()
        if len(lines) > self.MAX_LINES:
            lines = lines[-self.MAX_LINES :]
            content = "\n".join(lines)
            self.setPlainText(content)
        elif len(lines) + self.document().blockCount() > self.MAX_LINES:
            n = len(lines) + self.document().blockCount() - self.MAX_LINES
            tc = self.textCursor()
            tc.movePosition(QTextCursor.Start)
            tc.movePosition(QTextCursor.NextBlock, QTextCursor.KeepAnchor, n)
            tc.deleteChar()
            self.moveToLastLine()
            self.append(content)
        else:
            self.append(content)

    """ Override these in your subclass """

    def runCommand(self, cmd):
        self.showPrompt()

    def prompt(self, userstring=""):
        self.ps = f"{userstring}{self.prompt_caret}"
        return self.ps


class Terminal(CommandLineInterface):
    def __init__(self, ctrl):
        CommandLineInterface.__init__(self)
        self.setObjectName("Terminal")
        self.label = lambda: "Terminal"
        self.toolKey = "terminal"
        self.defaultLocation = "right"
        self.ctrl = ctrl
        self.childGenerator = None
        self.subprocess = SubprocessHandler()
        self.subprocess.resultsReady.connect(self.appendOutput)
        self.subprocess.finished.connect(self.showPrompt)
        self.ctrl.childThreads.append(self.subprocess)

        A = ActionEnum()
        # cat, cp, find, grep, ls, md5sum, mv, rm, touch, ztar
        A.print_files.addHandler(self.generatorWrapper(cat.main))
        A.copy_files.addHandler(self.generatorWrapper(cp.main))
        A.find_files.addHandler(self.generatorWrapper(find.main))
        A.search_in_files.addHandler(self.generatorWrapper(grep.main))
        A.list_files.addHandler(self.generatorWrapper(ls.main))
        A.md5sum_file.addHandler(self.generatorWrapper(md5sum.main))
        A.move_files.addHandler(self.generatorWrapper(mv.main))
        A.remove_files.addHandler(self.generatorWrapper(rm.main))
        A.create_files.addHandler(self.generatorWrapper(touch.main))
        #A.ztar.addHandler(self.ztar)

    def prompt(self):
        # TODO: should this be more user-configurable?
        cwd = os.getcwd()
        host = socket.gethostname()
        user = getpass.getuser()
        return CommandLineInterface.prompt(self, userstring=f"{user}@{host}:{cwd}")

    def logAction(self, message):
        tc = self.textCursor()
        tc.insertText(message)
        self.setTextCursor(tc)
        # logAction is only called when an action wasn't called from the CLI
        # but it's output may still be async
        # TODO: a finished signal for all actions
        self.showPrompt()

    def runCommand(self, cmdString):
        # TODO: Don't hang the GUI
        # TODO: handle raw mode applications (ie vim) (ANSI Escape codes)
        # TODO: shlex quote?
        # TODO: asyncio.create_subprocess_shell (use ProactorEventLoop on Windows)
        # TODO: add my basic unix tools
        # asyncio would have to be integrated with Qt's event loop somehow
        # alternative is subprocess.Popen and Popen.poll in a separate QThread
        args = shlex.split(cmdString)
        # first attempt in application Actions
        try:
            # convience syntax
            if args[0] in ActionEnum():
                cmdString = args[0] + "()"
            ActionDispatcher().cli(cmdString)
        except NameError:
            pass
        else:
            return
        # next check for a windows drive change
        if re.match("[a-zA-Z]:", args[0]):
            self.append("Change drive letters with cd [drive letter]:")
            self.showPrompt()
        # some basic shell commands
        elif args[0] == "cd":
            if len(args) == 1:
                path = os.path.expanduser("~")
            else:
                path = args[1]
            try:
                os.chdir(path)
            except Exception as e:
                self.appendOutput(str(e))
            self.showPrompt()
        elif args[0] == "cls":
            self.clear()
            self.showPrompt()
        # o/w try searching the os
        else:
            cmd = shutil.which(args[0])
            if cmd:
                args[0] = cmd
                self.subprocess.start(args)
            else:
                self.append(f"Command not found: {args[0]}")
                self.showPrompt()

    def createGeneratorHandler(self, generator, args=(), kwargs=()):
        if self.childGenerator:
            # TODO: handle running generators
            self.childGenerator.stopThread()
            self.ctrl.childThreads.remove(self.childGenerator)
        self.childGenerator = GeneratorOutputHandler(generator, args, kwargs)
        self.childGenerator.resultsReady.connect(self.appendOutput)
        self.childGenerator.finished.connect(self.showPrompt)
        self.childGenerator.start()
        self.ctrl.childThreads.append(self.childGenerator)

    def generatorWrapper(self, func):
        def _wrapper(*args, **kwargs):
            self.createGeneratorHandler(func, args, kwargs)
        return _wrapper
