""" A simple dock to pick a value from a generator """
from PySide2.QtWidgets import (
    QDockWidget,
    QTextEdit,
    QWidget,
    QVBoxLayout,
)
from PySide2.QtGui import (
    QTextCharFormat,
    QBrush,
    QColor,
)
from PySide2.QtCore import Qt

from source.fuzzysearch import FuzzySearch


# TODO: factor out to utility class
def makeFormat(foreground, italic=False, bold=False, underline=False, background=None):
    """create a QTextCharFormat based on some common attributes"""
    tcFormat = QTextCharFormat()
    tcFormat.setForeground(QBrush(QColor(foreground)))
    if background:
        tcFormat.setBackground(QBrush(QColor(background)))
    if italic:
        tcFormat.setFontItalic(True)
    if bold:
        tcFormat.setFontWeight(100)
    return tcFormat


class _SimplePicker(QWidget):
    PAGENUM = 20

    def __init__(self, ctrl, name, label):
        QWidget.__init__(self)
        self.setObjectName(name)  # been wanting to use that
        self.label = lambda: label
        self.toolKey = "picker"
        self.defaultLocation = "right"
        self.ctrl = ctrl
        # child widgets
        self.view = QTextEdit()
        # self.view.setFocusPolicy(Qt.NoFocus)
        self.view.setReadOnly(True)
        # self.cli = QLineEdit()
        self.lo = QVBoxLayout(self)
        self.lo.addWidget(self.view)
        # self.lo.addWidget(self.cli)
        self.view.keyPressEvent = self.processKey
        # styles
        self.highlightFormat = makeFormat("#111111", background="#cccccc")
        self.highlightSubtleFormat = makeFormat("#11aa11", background="#eeffee")
        self.subtleFormat = makeFormat("#aaffaa", background="#111111")
        self.normalFormat = makeFormat("#eeeeee", background="#111111")
        # setup child thread
        self.fzf = None
        self.selectedCallback = None
        # focus
        self.setFocusProxy(self.view)
        self.view.focusOutEvent = self.focusOutEvent

    def popup(self, generator, selectedCallback):
        assert callable(selectedCallback)
        self.selectedCallback = selectedCallback
        self.kickoffWalker(generator)

    def kickoffWalker(self, generator):
        if self.fzf:
            self.fzf.stopThread()
        self.fzf = FuzzySearch(generator)
        self.fzf.pageReady.connect(self.updateList)
        self.fzf.start()
        self.fzf.fetchPage(0, 20)

    def focusOutEvent(self, event):
        self.hide()
        self.fzf.stopThread()

    def closeEvent(self, event):
        if self.fzf:
            self.fzf.stopThread()

    def processKey(self, event):
        # QLineEdit.keyPressEvent(self.cli, event)
        key = event.key()
        # TODO: page up/down
        if key == Qt.Key_Up:
            self.fzf.selectPrevious()
        elif key == Qt.Key_Down:
            self.fzf.selectNext()
        elif key == Qt.Key_Return:
            # select
            self.selectedCallback(self.fzf.selected)
            self.fzf.stopThread()
            self.hide()

    def updateList(self, results):
        self.view.clear()
        tc = self.view.textCursor()
        if len(results) == 0:
            return
        for span, fn in results:
            if span:
                i, j = span
                if fn == self.fzf.selected:
                    tc.insertText(fn[:i], self.highlightFormat)
                    tc.insertText(fn[i:j], self.highlightSubtleFormat)
                    tc.insertText(fn[j:] + "\n", self.highlightFormat)
                else:
                    tc.insertText(fn[:i], self.normalFormat)
                    tc.insertText(fn[i:j], self.subtleFormat)
                    tc.insertText(fn[j:] + "\n", self.normalFormat)
                continue
            if fn == self.fzf.selected:
                tc.insertText(fn + "\n", self.highlightFormat)
            else:
                tc.insertText(fn + "\n", self.normalFormat)
        tc.insertText(
            f"{self.fzf.i}-{self.fzf.j}/{self.fzf.numFound}", self.normalFormat
        )
        self.view.setTextCursor(tc)


""" This is temporary on my way to converting my docks to tools
    and making "dock" just a ui layout concept
"""
from source.dockwidget import ToolDock


class SimplePickerDock(ToolDock):
    def __init__(self, parent, ctrl, name, label):
        child = _SimplePicker(ctrl, name, label)
        ToolDock.__init__(self, ctrl, parent, child)
        self.label = label
