import os
import sys
import subprocess

# used by settings.py
from string import Template

from source.basecls import Singleton
from source.action import ActionEnum
from source.logger import Logger


class _Settings:
    def __init__(self):
        self.exedir = (
            hasattr(sys, "_MEIPASS")
            and os.path.abspath(os.path.dirname(sys.executable))
            or os.path.abspath(os.path.dirname(sys.argv[0]))
        )
        self.base_paths = (self.exedir,)
        self.settings = {}
        # These are None so that an error is raised if you attempt
        # to use them before they are set
        self.stylesheet = None
        self.hilitestyles = None
        self.pythonexe = None
        for fn in ("python3", "python"):
            try:
                subprocess.run([fn, "--version"])
            except Exception as e:
                Logger.debug(f"Exception while detecting python version: {repr(e)}")
            else:
                self.pythonexe = fn
                Logger.info(f"Using python command: {fn}")
                break
        self.load_settings()
        self.load_macros()
        self.load_actions()

    def load_resource(self, filename, subdir="data"):
        """ returns a complete path to a specific file from the user directory
            searching through self.base_paths
        """
        for base in self.base_paths:
            p = os.path.join(base, subdir, filename)
            if os.path.exists(p):
                return p

    def load_resources(self, subdir="data"):
        """ returns a list of all the filesnames (complete paths) in a user 
            directory, including all paths in self.base_paths.
            does not recurse
        """
        for base in self.base_paths:
            d = os.path.join(base, subdir)
            for filename in os.listdir(d):
                p = os.path.join(d, filename)
                if not os.path.isdir(p):
                    yield p

    def load_user_script(self, filename, subdir="script", local_scope=None):
        if not local_scope:
            local_scope = {}
        try:
            exec(
                compile(
                    open(self.load_resource(filename, subdir)).read(), filename, "exec"
                ),
                globals(),
                local_scope,
            )
        except Exception as e:
            raise
        return local_scope

    def load_settings(self):
        # TODO: load from user folder first
        self.settings.update(self.load_user_script("settings.py", "data"))
        self.stylesheet = self.settings["style"]()

    def load_macros(self):
        scope = {"A": ActionEnum()}
        d = os.path.join("data", "macros")
        for filename in self.load_resources(subdir=d):
            filename = os.path.split(filename)[1]
            self.load_user_script(
                filename, subdir=d, local_scope=scope
            )
            for key in scope.keys():
                if key.startswith("_") or key == "A":
                    continue
                # basecls.AutoNamedMembers creates the Action
                setattr(ActionEnum(), key, None)
                getattr(ActionEnum(), key).addHandler(scope[key])

    def load_actions(self):
        """ adds: SHORTCUTS, MENUS, and ABBREVIATIONS
            also defines Action interfaces
        """
        self.settings.update(
            self.load_user_script(
                "actions.py", subdir="data", local_scope={"A": ActionEnum()}
            )
        )

    def get(self, attr):
        return self.settings[attr]


Settings = Singleton(_Settings)
