import os
import subprocess
from PySide2.QtWidgets import (
    QDockWidget,
    QListWidget,
    QListWidgetItem,
    QWidget,
    QVBoxLayout,
    QStackedLayout,
    QPushButton,
    QLabel,
)
from PySide2.QtCore import Qt

import source.vcs as vcs
from source.gui import MainGui


class ListItem(QListWidgetItem):
    def __init__(self, label, fn, status):
        QListWidgetItem.__init__(self, label)
        self.fn = fn
        self.status = status
        self.label = label

    def __lt__(self, other):
        return self.status > other.status or (
            self.status == other.status and self.label < other.label
        )


class _ProjectStatus(QWidget):
    def __init__(self, ctrl):
        QWidget.__init__(self)
        self.setObjectName("ProjectStatusDock")

        self.label = lambda: "Project Status"
        self.toolKey = "project"
        self.defaultLocation = "right"
        self.ctrl = ctrl

        self.fileList = QListWidget()
        self.fileList.setSortingEnabled(True)
        self.fileList.itemActivated.connect(self.openFile)

        self.initPage = QWidget()
        initLO = QVBoxLayout(self.initPage)
        initLO.addWidget(QLabel("Init a version control system"))
        avail = vcs.getAvailableVCS()
        if not avail:
            initLO.addWidget(QLabel("No version control systems are available"))
        self.initBtns = []
        for cmd in avail:
            btn = QPushButton(cmd)
            btn.clicked.connect(lambda: self.init(cmd))
            initLO.addWidget(btn)
            self.initBtns.append(btn)

        self.createPage = QWidget()
        createLO = QVBoxLayout(self.createPage)
        self.createBtn = QPushButton("Create a Project")
        self.createBtn.clicked.connect(self.ctrl.createProject)
        self.openBtn = QPushButton("Open a Project")
        self.openBtn.clicked.connect(self.ctrl.openProject)
        createLO.addWidget(self.createBtn)
        createLO.addWidget(self.openBtn)

        self.lo = QStackedLayout(self)
        self.lo.addWidget(self.fileList)
        self.lo.addWidget(self.initPage)
        self.lo.addWidget(self.createPage)
        # TODO: make changing focus cycle thru btns instead of returning to TextEdit

        # TODO: add setup CI action
        # TODO: if CI is setup show test & build status
        # TODO: allow adding/ignoring untracked files
        self.lo.setCurrentWidget(self.createPage)
        self.setWindowTitle("Project Status: None")

    def init(self, vcsType):
        if self.ctrl.project:
            vcs.init(vcsType, self.ctrl.project.path)
            self.ctrl.project.vcs = vcsType
            self.updateProject(self.ctrl.project)

    def openFile(self, item):
        fn = item.fn
        MainGui().open(fn)

    def updateProject(self, project):
        self.setWindowTitle(f"{project.name}: {project.path}")
        if not project.vcs:
            self.lo.setCurrentWidget(self.initPage)
            self.setFocusProxy(self.initPage)
        else:
            self.lo.setCurrentWidget(self.fileList)
            self.setFocusProxy(self.fileList)
            self.updateFileList(project)

    def updateFileList(self, project):
        if not project.vcs:
            return
        path = project.path
        self.fileList.clear()
        status = vcs.getStatus(project.vcs, project.path)
        tracked = vcs.listFiles(project.vcs, project.path)
        # add the untracked files
        for fn in status.keys():
            if not fn in tracked:
                label = status[fn] + fn[len(project.path) + 1 :]
                item = ListItem(label, fn, status[fn])
                self.fileList.addItem(item)
        # and the tracked files
        for fn in tracked:
            label = fn[len(project.path) + 1 :]
            if fn in status:
                label = status[fn] + label
                item = ListItem(label, fn, status[fn])
            else:
                item = ListItem(label, fn, "")
            self.fileList.addItem(item)


""" This is temporary on my way to converting my docks to tools
    and making "dock" just a ui layout concept
"""
from source.dockwidget import ToolDock


class ProjectStatus(ToolDock):
    def __init__(self, parent, ctrl):
        child = _ProjectStatus(ctrl)
        ToolDock.__init__(self, ctrl, parent, child)
        self.label = "Project Status"
