from PySide2.QtGui import (
    QSyntaxHighlighter,
    QTextCharFormat,
    QBrush,
    QColor,
    QTextBlockUserData,
)

from forked_pygments.token import Token, string_to_tokentype
from forked_pygments.lexers import PythonLexer

from source.resources import Settings


def makeFormat(foreground, italic=False, bold=False, underline=False, background=None):
    """create a QTextCharFormat based on some common attributes"""
    tcFormat = QTextCharFormat()
    tcFormat.setForeground(QBrush(QColor(foreground)))
    if background:
        tcFormat.setBackground(QBrush(QColor(background)))
    if italic:
        tcFormat.setFontItalic(True)
    if bold:
        tcFormat.setFontWeight(100)
    return tcFormat


class UserData(QTextBlockUserData):
    """Stores the highlighter state of the current textblock"""

    def __init__(self, stack):
        QTextBlockUserData.__init__(self)
        self.stack = stack


class Highlighter(QSyntaxHighlighter):
    """Pygments based highlighter class"""

    def __init__(self, doc):
        QSyntaxHighlighter.__init__(self, doc)
        self.lex = PythonLexer()
        self.setup_styles()

    def setup_styles(self):
        HILITESTYLES = Settings().get("HILITESTYLES")
        # WARNING: a typo on Token.XX not throw an error
        self.defaultStyle = makeFormat(**HILITESTYLES["Text"])
        self.styles = {}
        for tokenString, props in HILITESTYLES.items():
            self.styles[string_to_tokentype(tokenString)] = makeFormat(**props)

    def styleMatcher(self, token):
        match = None
        for tokenType in self.styles:
            if token in tokenType and (not match or len(tokenType) > len(match)):
                match = tokenType
        return match

    def highlightBlock(self, text):
        block = self.currentBlock()
        userdata = block.previous().userData()
        if not userdata or not userdata.stack:
            stack = ("root",)
        else:
            stack = userdata.stack
        for r in self.lex.get_tokens_unprocessed(text, stack):
            i, tt, v, stack = r
            match = self.styleMatcher(tt)
            if match:
                self.setFormat(i, len(v), self.styles[match])
            else:
                # This is here to fix the weird missing lines bug
                # that I noticed on some nested tuples but had
                # trouble finding a minimal example that reproduced
                # it's not related to block.setVisible
                # and setting the textCharFormat fixed it
                # it's unclear to me what happening
                self.setFormat(i, len(v), self.defaultStyle)
        self.setCurrentBlockUserData(UserData(stack))
