import os
import yaml
from dataclasses import asdict
from PySide2.QtWidgets import (
    QMainWindow,
    QWidget,
    QLabel,
    QLineEdit,
    QStackedLayout,
    QFileDialog,
    QInputDialog,
    QMessageBox,
    QDockWidget,
    QCompleter,
)
from PySide2.QtGui import QKeySequence, QIcon
from PySide2.QtCore import Qt, QByteArray

from source.basecls import Singleton
from source.actionbar import ActionBar
from source.resources import Settings
from source.projects import saveProject, loadProject, Project
from source.action import ActionEnum
from source.toolregistry import ToolRegistry, ToolKeys
from source.watcher import Watcher


class MyFileDialog(QFileDialog):
    def __init__(self, parent, caption, handler, openKind=False):
        QFileDialog.__init__(self, parent, caption)
        self.setModal(True)
        self.setOption(
            QFileDialog.DontUseNativeDialog
        )  # This has no affect (its always widget based AFAIK)
        if openKind:
            self.setFileMode(QFileDialog.ExistingFile)
            self.setAcceptMode(QFileDialog.AcceptOpen)
        else:
            self.setFileMode(QFileDialog.AnyFile)
            self.setAcceptMode(QFileDialog.AcceptSave)
        self.accepted.connect(handler)

    def done(self, *args):
        QFileDialog.done(self, *args)
        # seems to fix misplaced focus bug
        for c in self.children():
            if isinstance(c, QCompleter):
                c.popup().hide()

    def showDialog(self):
        self.setDirectory(os.getcwd())
        return self.open()


class _MainGui(QMainWindow):
    def __init__(self):
        QMainWindow.__init__(self)
        self.setObjectName("MainGui")
        self.central = QWidget()
        self.central.setObjectName("Central")
        self.lo = QStackedLayout(self.central)
        self.setCentralWidget(self.central)
        self.project = None
        ToolReg = ToolRegistry()
        # DOCKS
        """ TODO: refactor
        self.projectStatus = ProjectStatus(self, ctrl)
        self.projectStatus.hide()
        self.autocompletePicker = SimplePickerDock(
            self, ctrl, "AutocompletePicker", "Autocomplete Suggestions"
        )
        self.addDockWidget(Qt.RightDockWidgetArea, self.autocompletePicker)
        self.autocompletePicker.hide()
        """
        # ACTION HANDLERS
        A = ActionEnum()
        A.new_file.addHandler(self.new)
        A.open_file.addHandler(self.openFile)
        A.save_file.addHandler(self.save)
        A.save_as_file.addHandler(self.showSaveAsDialog)
        A.close_file.addHandler(self.closeTool)
        A.remove_file.addHandler(self.remove)
        A.quit.addHandler(self.quit)
        # TODO: pass to docks in focus, not just the current tool
        A.copy_selection.addHandler(self.copy)
        A.cut_selection.addHandler(self.cut)
        A.paste_from_clipboard.addHandler(self.paste)
        A.suggest_autocomplete.addHandler(lambda: ToolReg.onTop().autocomplete())
        A.show_find_dock.addHandler(self.search)
        A.show_replace_dock.addHandler(self.replace)
        A.focus_on_main_tool.addHandler(lambda: ToolReg.onTop().setFocus())
        A.focus_next_tool.addHandler(ToolReg.nextTool)
        A.focus_previous_tool.addHandler(ToolReg.backTool)
        A.show_pyhelp_dock.addHandler(self.trigger_help)
        A.show_open_files_dock.addHandler(lambda: ToolReg.toggleTool(ToolKeys.fileList))
        # A.show_project_dock.addHandler(lambda: self.toggleTool(ToolKeys.projectStatus))
        A.show_terminal_dock.addHandler(lambda: ToolReg.toggleTool(ToolKeys.terminal))
        A.show_file_manager_dock.addHandler(
            lambda: ToolReg.toggleTool(ToolKeys.fileManager)
        )
        A.toggle_between_focus_dock.addHandler(
            lambda: ToolReg.toggleToolBetweenFocusAndDock()
        )
        A.grow_dock.addHandler(self.biggenDock)
        A.shrink_dock.addHandler(self.smallenDock)
        A.open_settings_file.addHandler(
            lambda: self.open(Settings().load_resource("settings.py"))
        )
        A.reload_settings_file.addHandler(self.setStyle)

        MENUS = Settings().get("MENUS")
        self.actionBar = ActionBar(MENUS)
        self.setMenuBar(self.actionBar)

        self.toolStatus = QLabel()
        self.statusBar().addPermanentWidget(self.toolStatus)

        self.openFileDialog = MyFileDialog(
            self,
            "Open...",
            lambda: self.open(self.openFileDialog.selectedFiles()[0]),
            openKind=True,
        )
        self.saveAsFileDialog = MyFileDialog(
            self,
            "Save As...",
            lambda: self.saveAs(self.saveAsFileDialog.selectedFiles()[0]),
        )

        # REMAINING SETUP
        self.displayUserMessage("Ready")

        # STYLING
        self.setStyle(reload=False)

        self.setWindowIcon(QIcon(Settings().load_resource("workbench.png")))
        if os.name == "nt":
            import ctypes

            myappid = "MyOrg.Mygui.1.0.0"  # Arbitrary
            ctypes.windll.shell32.SetCurrentProcessExplicitAppUserModelID(myappid)

    def spawnDefaultTools(self):
        ToolReg = ToolRegistry()
        self.fileList = ToolReg.spawn(ToolKeys.fileList)
        ToolReg.hide(self.fileList)
        self.terminal = ToolReg.spawn(ToolKeys.terminal)
        ToolReg.hide(self.terminal)
        self.pyhelper = ToolReg.spawn(ToolKeys.pythonHelper)
        ToolReg.hide(self.pyhelper)
        self.searchBox = ToolReg.spawn(ToolKeys.search)
        ToolReg.hide(self.searchBox)
        self.replaceBox = ToolReg.spawn(ToolKeys.replace)
        ToolReg.hide(self.replaceBox)
        self.fileManager = ToolReg.spawn(ToolKeys.fileManager)
        ToolReg.hide(self.fileManager)
        self.actionLog = ToolReg.spawn(ToolKeys.actionLog)
        ToolReg.hide(self.actionLog)
        ToolReg.spawn(ToolKeys.editor)

    def new(self):
        current = ToolRegistry().onTop()
        if current.filename() or current.isUnsaved():
            ToolRegistry().spawn(ToolKeys.editor)
            self.fileList.addItem(current)

    def openFile(self, fn=None):
        if not fn:
            self.showOpenDialog()
        else:
            self.open(fn)

    def showOpenDialog(self):
        self.openFileDialog.showDialog()

    def open(self, filename):
        if not os.path.exists(filename):
            self.displayUserMessage(f"File {filename} does not exist.")
        currentTool = ToolRegistry().onTop()
        if not currentTool or currentTool.filename() or currentTool.isUnsaved():
            ToolRegistry().spawn(ToolKeys.editor)
            currentTool = ToolRegistry().onTop()
        else:
            self.fileList.removeItem(currentTool)
        currentTool.load(filename)
        Watcher().addPath(filename)
        self.fileList.addItem(currentTool)

    def save(self):
        currentTool = ToolRegistry().onTop()
        if not currentTool.filename():
            self.saveAs()
        else:
            currentTool.save()

    def showSaveAsDialog(self):
        self.saveAsFileDialog.showDialog()

    def saveAs(self, filename):
        currentTool = ToolRegistry().onTop()
        self.fileList.removeItem(currentTool)
        currentTool.saveAs(filename)
        self.fileList.addItem(currentTool)

    def closeTool(self):
        # TODO: check for unsaved changes
        toClose = ToolRegistry().onTop()
        ToolRegistry().remove(toClose)
        toClose.close()

    def remove(self):
        # removes a file from the filesystem
        if self.fileManager.hasFocus():
            self.fileManager.remove()

    def quit(self):
        # TODO: check for unsaved files
        if self.project:
            toolStates = []
            for tool in ToolRegistry().allTools():
                # TODO: Need special handling here if .project.yaml is the open file
                toolStates.append(tool.saveState())
            self.project.savedStates = toolStates
            self.project.geometry = self.saveGeometry().toBase64().data().decode()
            self.project.docks = self.saveState().toBase64().data().decode()
            saveProject(self.project)
        self.close()

    def newWeb(self):
        # Open a WWW Tool if there isn't (a blank) one. similar to "new"
        pass

    def get(self):
        # get the page, given the URL in the www address bar
        # if the page is not in the cache: request, cache the page
        # display from cache
        pass

    def getLatest(self):
        # request latest copy of page from server
        # cache
        # self.get
        pass

    def copy(self):
        ToolRegistry().onTop().copy()

    def cut(self):
        ToolRegistry().onTop().cut()

    def paste(self):
        ToolRegistry().onTop().paste()

    def search(self):
        ToolRegistry().toggleTool(ToolKeys.search)

    def replace(self):
        ToolRegistry().toggleTool(ToolKeys.replace)

    def displayUserMessage(self, message):
        self.statusBar().showMessage(message, 5000)

    def updateTitleBar(self, message):
        self.setWindowTitle(message)

    def trigger_help(self):
        if ToolRegistry().onTop().hasFocus():
            prompt = ToolRegistry().onTop().underCursor()
            if prompt:
                self.pyhelper.prime(prompt)
        ToolRegistry().toggleTool(ToolKeys.pythonHelper)

    def biggenDock(self, increment=10):
        w = None
        for d in ToolRegistry().allDocks():
            if d.hasFocus():
                w = d
                break
        if w is None:
            return
        if self.dockWidgetArea(w) in (Qt.LeftDockWidgetArea, Qt.RightDockWidgetArea):
            size = w.size().width() + increment
            orientation = Qt.Horizontal
        else:
            size = w.size().height() + increment
            orientation = Qt.Vertical
        self.resizeDocks([w], [size], orientation)

    def smallenDock(self):
        self.biggenDock(increment=-10)

    def setStyle(self, reload=True):
        if reload:
            Settings().load_settings()
        self.setStyleSheet("")
        STYLESHEET = Settings().stylesheet
        self.setStyleSheet(STYLESHEET)
        self.actionBar.reloadStyle(STYLESHEET)

    def log(self, message):
        """write a log of an action"""
        self.actionLog.logAction(message)

MainGui = Singleton(_MainGui)
