""" fuzzysearch.py
    An abstracted interface for fuzzy search style pick lists.
    Allowing background thread to add items while the user interacts (filter/sort/select).
    Two things need to be run inside the child thread: 
        Generating the list.
        Filtering, sorting and/or searching the list.
    Additionally the child thread must support two states during which the user can still interact:
        While the list is being generated.
        When the list is complete.

    Parent creates a generator function and child thread, passing the generator.
    Child creates the model.
    While generator is running it populates the model and pauses on a defined interval to check for user interaction and emit results.
    Child handles sorting, filtering, searching, pagination.
    All parent/child interaction requires signals/slots for cross thread communincation (unless I want to switch to python native threads and queues).
    But signals/slots don't work until the event loop is started (ie while generator is running) so an alternative method of signaling is required (ie set flags).
    If the generator changes the parent terminates the child and starts over.

    Should the model use sort and then search or search and then sort?

    Use SortedKeyList and allowing passing in the key function.
    Then filter.
    How to return results before filtering is complete? That might not be necessary given that filtering happens on in-memory structure. 

    QThread: Initially I wanted to implement this entirely with threading and queue libraries however that would require polling the results queue (and command queue once the generator was complete) in order to integrate with the Qt event loop. In order to eliminate polling a ready signal needs to be added for each queue.

"""

import os
import re
import time
import queue
from enum import Enum

# import threading
from sortedcontainers import SortedKeyList
from PySide2.QtCore import Signal, QThread, QObject


def exclude_hidden(path):
    return re.match("^[^.].*", path) is not None


# example generator
def fileWalker(cwd, relpath=True, prefilter=exclude_hidden):
    """cwd: current working directory
    relpath: True indicates that returned paths are relative to cwd
    prefilter: a callable that must return true for the path to be accepted
    """

    def generator():
        for root, dirs, files in os.walk(cwd):
            if prefilter:
                temp_dirs = [d for d in dirs if prefilter(d)]
                dirs[:] = temp_dirs
                files = [f for f in files if prefilter(f)]
            for f in dirs + files:
                path = os.path.join(root, f)
                if relpath:
                    path = os.path.relpath(path, cwd)
                yield path

    return generator


class FilterListModel:
    """A sorted list implementating pagination, sort, and filter."""

    def __init__(self):
        self.filterRE = None
        self.raw = SortedKeyList()
        self.filtered = SortedKeyList()

    def extend(self, entries):
        self.raw.update(entries)
        if self.filterRE:
            self._extendFiltered(entries)

    def _extendFiltered(self, entries):
        for entry in entries:
            match = self.filterRE.search(entry)
            if match:
                s = match.span()
                self.filtered.add((s, entry))

    def changeFilter(self, filterRE):
        self.filterRE = re.compile(filterRE)
        self.filtered = SortedKeyList()
        # TODO: handle returning some results if below is slow
        # if you make this async self.remove() needs to be rewritten
        self._extendFiltered(self.raw)

    def remove(self, entry):
        self.raw.remove(entry)
        # how do I remove from the filtered list?
        # recompute the match?
        # maintain a separate table and look up?
        # turns out remove is O(log(n)), so for now I'll just compute it
        # if the above rebuilding of self.filtered becomes async than the below
        # will not work (sometimes).
        if self.filterRE:
            match = self.filterRE.search(entry)
            if match:
                s = match.span()
                self.filtered.remove((s, entry))

    def add(self, entry):
        self.extend([entry])

    def page(self, i, j):
        self.i, self.j = i, j
        # right now this returns a list. If you change to an iterator the page caching
        # in FuzzySearchThread will break
        if self.filterRE:
            return [(s, m) for s, m in self.filtered.islice(i, j)]
        else:
            return [(None, m) for m in self.raw.islice(i, j)]

    def index(self, entry):
        if self.filterRE:
            match = self.filterRE.search(entry)
            if match:
                s = match.span()
                return self.filtered.index((s, entry))
            # returns None if the current filter does not match the entry
            return None
        else:
            return self.raw.index(entry)

    def __len__(self):
        if self.filterRE:
            return len(self.filtered)
        else:
            return len(self.raw)

    def __getitem__(self, index):
        if self.filterRE:
            return self.filtered[index][1]
        else:
            return self.raw[index]


class COMMANDS(Enum):
    STOP = 1
    CHANGEFILTER = 2
    FETCHPAGE = 3
    ADD = 4
    REMOVE = 5
    PREVIOUS = 6
    NEXT = 7


class FuzzySearchThread(QThread):
    BUFFER_TIME = 0.1

    def __init__(self, generator, model, resultsHandler, commandSignal):
        QThread.__init__(self)
        self.generator = generator
        self.model = model
        self.handler = resultsHandler
        self.commandSignal = commandSignal
        self.commandQ = queue.Queue()
        self.resultsQ = queue.Queue()
        self.stop = False
        # for storing the last requested page
        self.i = 0
        self.j = 0
        self.cachedPage = None
        self.selected = None
        self.numFound = 0

    def run(self):
        # Phase 1: loads the data, check for commands periodically
        timer = time.time()
        buf = []
        for entry in self.generator():
            buf.append(entry)
            now = time.time()
            if now - timer > self.BUFFER_TIME:
                self.model.extend(buf)
                self.numFound = len(self.model)
                self.processCommands()
                if self.stop:
                    return
                # emit pageReady if last slice results changed
                self.sendPage()
                timer = now
                buf = []
        # handle left over results and commands since last timer expiration
        self.model.extend(buf)
        self.processCommands()
        if self.stop:
            return
        # Phase 2: connect the signal handler and start the event loop
        self.commandSignal.connect(self.processCommands)
        self.exec_()

    def processCommands(self):
        if self.isInterruptionRequested():
            self.stop = True
            self.exit()  # Stops event processing and returns here
        while not self.commandQ.empty():
            command, arg = self.commandQ.get()
            if command == COMMANDS.STOP:
                self.stop = True
                self.exit()
            elif command == COMMANDS.CHANGEFILTER:
                self.model.changeFilter(arg)
                self.numFound = len(self.model)
                self.sendPage()
            elif command == COMMANDS.FETCHPAGE:
                self.i, self.j = arg
                self.sendPage()
            elif command == COMMANDS.ADD:
                self.model.add(arg)
                self.numFound = len(self.model)
                self.sendPage()
            elif command == COMMANDS.REMOVE:
                self.model.remove(arg)
                self.numFound = len(self.model)
                self.sendPage()
            elif command == COMMANDS.PREVIOUS:
                if self.selected:
                    idx = self.model.index(self.selected)
                    if idx is None:
                        return
                    if idx > 0:
                        idx -= 1
                        self.selected = self.model[idx]
                        if idx < self.i:
                            self.i, self.j = self.i - 1, self.j - 1
                        self.sendPage()
            elif command == COMMANDS.NEXT:
                if self.selected:
                    idx = self.model.index(self.selected)
                    if idx is None:
                        return
                    if idx < len(self.model) - 1:
                        idx += 1
                        self.selected = self.model[idx]
                        if idx >= self.j:
                            self.i, self.j = self.i + 1, self.j + 1
                        self.sendPage()

    def sendPage(self):
        page = self.model.page(self.i, self.j)
        # if page == self.cachedPage:
        #    return
        self.cachedPage = page
        # figure out if the selection isn't on this page
        if self.selected:
            idx = self.model.index(self.selected)
            if idx is None or idx < self.i or idx > self.j:
                self.selected = None
        # reset the selection to the first entry
        if self.selected is None:
            if len(self.cachedPage) > 0:
                self.selected = self.cachedPage[0][1]
            else:
                self.selected = None
        self.handler(self.cachedPage)


class FuzzySearch(QObject):
    """Interface to FuzzySearch module"""

    pageReady = Signal(list)
    commandReady = Signal()

    def __init__(self, generator, modelClass=None):
        QObject.__init__(self)
        if modelClass is None:
            modelClass = FilterListModel
        self.modelClass = modelClass
        self._createChild(generator)

    def _createChild(self, generator):
        self.child = FuzzySearchThread(
            generator, self.modelClass(), self._emitResults, self.commandReady
        )

    def _emitResults(self, results):
        self.pageReady.emit(results)

    @property
    def i(self):
        return self.child.i

    @property
    def j(self):
        return self.child.j

    @property
    def selected(self):
        return self.child.selected

    @property
    def numFound(self):
        return self.child.numFound

    def start(self):
        self.child.start()

    def changeGenerator(self, generator):
        # change the generator
        self.stopThread()
        self._createChild(generator)

    def changeSearch(self, term):
        # change the search string
        self.child.commandQ.put((COMMANDS.CHANGEFILTER, term))
        self.commandReady.emit()

    def fetchPage(self, i, j):
        # update the index of the result set
        self.child.commandQ.put((COMMANDS.FETCHPAGE, (i, j)))
        self.commandReady.emit()

    def stopThread(self):
        # stop the child thread
        self.child.commandQ.put((COMMANDS.STOP, None))
        self.commandReady.emit()
        self.child.wait()

    def addEntry(self, entry):
        # Adds an entry to the list from outside the generator
        self.child.commandQ.put((COMMANDS.ADD, entry))
        self.commandReady.emit()

    def removeEntry(self, entry):
        # removes an entry from the list from outside the generator
        self.child.commandQ.put((COMMANDS.REMOVE, entry))
        self.commandReady.emit()

    def selectPrevious(self):
        self.child.commandQ.put((COMMANDS.PREVIOUS, None))
        self.commandReady.emit()

    def selectNext(self):
        self.child.commandQ.put((COMMANDS.NEXT, None))
        self.commandReady.emit()
