""" fsman3.py - filesystem manager: keyboard driven, noun-verb interface

    # UI definition
    [/] default behavior: list CWD, filter by startswith, then contains.
    matched string is shown by text color.
   
    [/] startswith matches are displayed first, followed by contains. 
    [/] The *best* match is displayed first with highlight (selected match).
    [/] show matched in selected
    [/] if no matches, no selection
    [/] "new file" indication   

    [/] enter: default action = open or chdir.
    [ ] shift-enter: secondary action: chdir to parent of file.

    [/] Up/Down arrows select next/previous match.
    [ ] Ctrl-A: Selects all matches. Used for commands that operate on multiple files.
    [ ] Page Up/Down scrolls.
    [/] Escape cancels current filter. 

    
    View commands: 
        [ ] Toggle long (detailed)/wide listing.
        [ ] Enable recursion, shows/filters all files.    

    [ ] mkdir and
    [/] mkfile always operate on a single new filename. Must have NO exact matches.

    Commands that can operate on N files will use the currently selected match unless Select All
    was triggered first.
    [ ] copy and
    [ ] move open a second "destination" pane in CWD, CLI clears to select destination folder.
    
    [/] Remove requires confirmation.
    [ ] RecurseRemove is separate command.
    [ ] ztar: opens 2nd window to name/place a tar.gz with selected contents. 
    
    # find in files is a separate widget
    [ ] grep in filtered files opens a 2nd pane for results. CLI clears to allow entering search string. 

    Tags is a separate widget
    [ ] search/grok: 2nd pane for results. Searches selected files/folders for tags in CLI.
    ASIDE: grep/grok can be the same command with a prefix for tags (like #hashtags) but tracking 
    which tags have been used is important for assigning consistent tags.
    [ ] tag: opens 2nd pane to select tags for file. 

    [ ] Double escape cancels op. Shift-Escape cancels op but saves CLI contents.

    Marks is a separate widget
    [ ] show marks: changes display to saved bookmarks.
    [ ] save mark: saves the CWD to the entered bookmark name.
    [ ] jump: jumps to the entered bookmark
    [ ] close marks: returns to filesystem view (double escape)
  
"""


import os
import re
import math
import queue
import shlex
import shutil
import time
from sortedcontainers import SortedKeyList

from PySide2.QtWidgets import (
    QDockWidget,
    QTextEdit,
    QLineEdit,
    QWidget,
    QVBoxLayout,
    QMessageBox,
)
from PySide2.QtGui import (
    QTextCharFormat,
    QBrush,
    QColor,
)

from PySide2.QtCore import Qt, QThread, Signal, Slot
from source.fuzzysearch import FuzzySearch, fileWalker
from source.gui import MainGui


def makeFormat(foreground, italic=False, bold=False, underline=False, background=None):
    """create a QTextCharFormat based on some common attributes"""
    tcFormat = QTextCharFormat()
    tcFormat.setForeground(QBrush(QColor(foreground)))
    if background:
        tcFormat.setBackground(QBrush(QColor(background)))
    if italic:
        tcFormat.setFontItalic(True)
    if bold:
        tcFormat.setFontWeight(100)
    return tcFormat


class FileManager(QWidget):
    PAGENUM = 20

    def __init__(self, ctrl):
        # setup dock
        QWidget.__init__(self)
        self.setObjectName("FileManager")
        self.label = lambda: "File Manager"
        self.toolKey = "filemanager"
        self.defaultLocation = "right"
        self.ctrl = ctrl
        # child widgets
        self.view = QTextEdit()
        self.view.setFocusPolicy(Qt.NoFocus)
        self.cli = QLineEdit()
        self.lo = QVBoxLayout(self)
        self.lo.addWidget(self.view)
        self.lo.addWidget(self.cli)
        self.cli.keyPressEvent = self.processKey
        # styles
        self.highlightFormat = makeFormat("#111111", background="#cccccc")
        self.highlightSubtleFormat = makeFormat("#11aa11", background="#eeffee")
        self.subtleFormat = makeFormat("#aaffaa", background="#111111")
        self.normalFormat = makeFormat("#eeeeee", background="#111111")
        # TODO: colors for directories, executables, archives (tar, gz, zip)
        self.directoryFormat = makeFormat("#aaffff", background="#111111")
        self.executableFormat = makeFormat("#ffaaaa", background="#111111")
        self.archiveFormat = makeFormat("#aaffaa", background="#111111")
        # TODO: emojii icons for filetypes?
        # setup child thread
        self.fzf = None
        self.kickoffWalker()
        # focus
        self.setFocusProxy(self.cli)
        self.setTitle()

    def setTitle(self):
        self.setWindowTitle(f"File Manager: {os.getcwd()}")

    def hide(self):
        # TODO: hide DOCK?
        pass

    def kickoffWalker(self):
        if self.fzf:
            self.fzf.stopThread()
            self.ctrl.childThreads.remove(self.fzf)
        self.fzf = FuzzySearch(fileWalker(os.getcwd()))
        self.ctrl.childThreads.append(self.fzf)
        self.fzf.pageReady.connect(self.updateList)
        self.fzf.start()
        self.fzf.fetchPage(0, 20)

    def closeEvent(self, event):
        # This isn't getting called on close
        if self.fzf:
            self.fzf.stopThread()

    def processKey(self, event):
        QLineEdit.keyPressEvent(self.cli, event)
        key = event.key()
        if key == Qt.Key_Return:
            self.processCommand()
        # TODO: page up/down
        elif key == Qt.Key_Up:
            self.fzf.selectPrevious()
        elif key == Qt.Key_Down:
            self.fzf.selectNext()
        elif key == Qt.Key_Escape:
            # TODO: escape already moves focus to tool so can't be used here
            self.cli.clear()
        else:
            self.fzf.changeSearch(self.cli.text())

    def processCommand(self):
        text = self.cli.text()
        args = shlex.split(text)
        self.cli.clear()
        cmd = ""
        if len(args) > 0:
            cmd = args[0].lower()
        # TODO Make these into menu actions
        if cmd == "cd":
            if len(args) == 1:
                os.chdir(os.path.expanduser("~"))
            else:
                os.chdir(args[1])
            self.setTitle()
            self.kickoffWalker()
        elif cmd == "..":
            os.chdir("..")
            self.setTitle()
            self.kickoffWalker()
        else:
            fn = self.fzf.selected
            if fn is None:
                fn = text
                if not fn:
                    return
                if not os.path.exists(fn):
                    open(fn, "w").write("")
                    # TODO: opening tools selected by file type
                    MainGui().open(fn)
                    self.fzf.addEntry(fn)
                    self.hide()
            elif os.path.isdir(fn):
                os.chdir(fn)
                self.setTitle()
                self.kickoffWalker()
            else:
                MainGui().open(fn)
                self.hide()

    def updateList(self, results):
        self.view.clear()
        tc = self.view.textCursor()
        cmd = self.cli.text()
        if len(results) == 0 and len(cmd) > 0:
            tc.insertText(cmd + " (new file)\n", self.highlightFormat)
            return
        for span, fn in results:
            if span:
                i, j = span
                if fn == self.fzf.selected:
                    tc.insertText(fn[:i], self.highlightFormat)
                    tc.insertText(fn[i:j], self.highlightSubtleFormat)
                    tc.insertText(fn[j:] + "\n", self.highlightFormat)
                else:
                    tc.insertText(fn[:i], self.normalFormat)
                    tc.insertText(fn[i:j], self.subtleFormat)
                    tc.insertText(fn[j:] + "\n", self.normalFormat)
                continue
            if fn == self.fzf.selected:
                tc.insertText(fn + "\n", self.highlightFormat)
            else:
                tc.insertText(fn + "\n", self.normalFormat)
        tc.insertText(
            f"{self.fzf.i}-{self.fzf.j}/{self.fzf.numFound}", self.normalFormat
        )
        self.view.setTextCursor(tc)

    def remove(self):
        if self.fzf.selected:
            if (
                QMessageBox.question(
                    MainGui(),
                    "Confirm",
                    f"Do you want to delete {self.fzf.selected}?",
                )
                == QMessageBox.StandardButton.Yes
            ):
                os.remove(self.fzf.selected)
                self.fzf.remove(self.fzf.selected)
                # TODO close tool with file open
