import re
import math
import shlex
import subprocess
from PySide2.QtWidgets import QTextEdit, QHBoxLayout, QWidget
from PySide2.QtGui import (
    QTextCursor,
    QTextDocument,
    QPainter,
    QPen,
    QBrush,
    QColor,
    QKeySequence,
)
from PySide2.QtCore import Qt, QRegularExpression, QSizeF, QRect, QPoint

from source.highlighter import Highlighter
from source.action import ActionEnum, InFocus, ActionDispatcher
from source.theforce import Lightsaber
from source.watcher import Watcher
from source.gui import MainGui
from source.logger import Logger


class Margin(QTextEdit):
    """The left hand margin, shows line numbers"""

    def __init__(self):
        QTextEdit.__init__(self)
        self.setFixedWidth(40)
        self.setVerticalScrollBarPolicy(Qt.ScrollBarAlwaysOff)
        self.setHorizontalScrollBarPolicy(Qt.ScrollBarAlwaysOff)
        self.setReadOnly(True)
        self.setFocusPolicy(Qt.NoFocus)
        self.wheelHandler = None

    def wheelEvent(self, event):
        """Oddly, despite having a NoFocus policy this widget can
        still receive scroll wheel events.
        """
        if callable(self.wheelHandler):
            self.wheelHandler(event)


class OverviewCursor(QWidget):
    def __init__(self, parent):
        QWidget.__init__(self, parent)
        self.setAttribute(Qt.WA_TransparentForMouseEvents)
        self.starty = 0
        self.endy = 100

    def paintEvent(self, event):
        painter = QPainter()
        painter.begin(self)
        painter.setPen(QPen(QColor(255, 0, 0)))
        painter.setBrush(QBrush(QColor(0, 0, 0, 0)))
        w = self.parent().size().width() - 1
        painter.drawRect(QRect(0, self.starty, w, self.endy - self.starty))
        painter.end()

    def updatePos(self, starty, endy):
        self.starty = starty
        self.endy = endy
        self.update()


class Overview(QTextEdit):
    """Provides a single screen view of the current file"""

    FIXEDWIDTH = 100

    def __init__(self):
        QTextEdit.__init__(self)
        self.setFixedWidth(Overview.FIXEDWIDTH)
        self.setVerticalScrollBarPolicy(Qt.ScrollBarAlwaysOff)
        self.setHorizontalScrollBarPolicy(Qt.ScrollBarAlwaysOff)
        self.setReadOnly(True)
        self.setFocusPolicy(Qt.NoFocus)
        self.setLineWrapMode(QTextEdit.NoWrap)
        # TODO: pull the height from the widget size
        self.document().setPageSize(QSizeF(0, 1000))
        self.cursor = OverviewCursor(self)
        self.cursor.raise_()
        self.wheelHandler = None

    def resizeEvent(self, event):
        QTextEdit.resizeEvent(self, event)
        self.cursor.resize(self.size())

    def load(self, filename):
        self.setPlainText(open(filename).read())
        if filename.endswith(".py"):
            self.hl = Highlighter(self.document())
        # 0.1 crashes my computer
        for f in (4, 3, 2, 1, 0.5):
            # self.setStyleSheet(f"font-size: {f}pt")
            self.zoomOut(f)  # I did not expect that to be equivalent
            if self.document().pageCount() <= 1:
                break

    def wheelEvent(self, event):
        if callable(self.wheelHandler):
            self.wheelHandler(event)

    def updateCursor(self, startBlockNumber, endBlockNumber):
        """updates the location of view frame"""
        doc = self.document()
        startBlock = doc.findBlockByNumber(startBlockNumber)
        endBlock = doc.findBlockByNumber(endBlockNumber)
        starty = self.cursorRect(QTextCursor(startBlock)).top()
        endy = self.cursorRect(QTextCursor(endBlock)).bottom()
        # what happens if a cursor isn't visible in the viewport?
        self.cursor.updatePos(starty, endy)


class Marks(QTextEdit):
    """Tracks quick marks"""

    def __init__(self):
        QTextEdit.__init__(self)
        self.setFixedWidth(20)
        self.setVerticalScrollBarPolicy(Qt.ScrollBarAlwaysOff)
        self.setHorizontalScrollBarPolicy(Qt.ScrollBarAlwaysOff)
        self.setReadOnly(True)
        self.setFocusPolicy(Qt.NoFocus)


class EditorArea(QTextEdit):
    def __init__(self, ctrl):
        QTextEdit.__init__(self)
        self.ctrl = ctrl
        self.mode = "insert"
        self.autoIndentRE = ".+:$"
        self.tab = "    "
        self.newline = "\n"
        self.message = ""
        self.cursorPositionChanged.connect(self._updateStatus)
        # TODO: paren matching
        # TODO: autosave (to tempfile if no repo, to actual file if repo is init)
        self.new()

    def _updateStatus(self):
        tc = self.textCursor()
        row = tc.blockNumber()
        col = tc.positionInBlock()
        MainGui().toolStatus.setText(f"({row},{col})")

    def insertFromMimeData(self, source):
        tc = self.textCursor()
        tc.insertText(source.text())

    def setFilename(self, filename):
        self.filename = filename
        self.document().setModified(False)
        self.message = ""
        self.updateTitleBar()

    def load(self, filename):
        # TODO: implement swap file
        self.setPlainText(open(filename).read())
        if filename.endswith(".py"):
            self.hl = Highlighter(self.document())
        self.setFilename(filename)

    def save(self):
        if self.filename:
            open(self.filename, "w").write(self.toPlainText())
            if self.filename.endswith(".py"):
                result = Lightsaber().compile(self.filename)
                if result.error:
                    MainGui().displayUserMessage(
                        f"{self.filename} failed to compile."
                    )
                    MainGui().terminal.appendOutput(result.message)
                    MainGui().toggleDock(self.ctrl.gui.terminal)
                    return
                result = Lightsaber().autoformat(self.filename)
                if result.error:
                    MainGui().displayUserMessage(
                        f"{self.filename} failed to autoformat."
                    )
                    MainGui().terminal.appendOutput(result.message)
                    MainGui().toggleDock(self.ctrl.gui.terminal)
                    return
                MainGui().displayUserMessage(f"{result.message}")
                # If autoformat ran you need to reload the file
                # and try to preserve the cursor location
                pos = self.textCursor().position()
                scrollPos = self.verticalScrollBar().value()
                self.load(self.filename)
                tc = self.textCursor()
                tc.setPosition(pos)
                self.setTextCursor(tc)
                self.verticalScrollBar().setValue(scrollPos)
                # Lightsaber().lint(self.filename)

    def saveAs(self, filename):
        self.setFilename(filename)
        self.save()

    def new(self):
        self.setFilename(None)
        self.setPlainText("")

    def search(self, exp, direction=None, repeat=False, flags=None):
        if repeat:
            tc = self.textCursor()
            tc.setPosition(tc.anchor())
            # if you don't do this it searches from the end of the current selection
            # messing up search as you type
            self.setTextCursor(tc)
        regExp = QRegularExpression(exp)
        secondPos = self.document().begin().position()
        if direction == "back":
            flags |= QTextDocument.FindBackward
            secondPos = self.document().end().position()
        if flags:
            found = self.find(QRegularExpression(exp), flags)
        else:
            found = self.find(QRegularExpression(exp))
        if not found:
            # try again with wrapping
            if flags:
                tc = self.document().find(regExp, secondPos, flags)
            else:
                tc = self.document().find(regExp, secondPos)
            if tc.hasSelection():
                self.setTextCursor(tc)
                found = True
        return found

    def replace(self, exp):
        """replaces the current selection with the exp, performing group substitution"""
        tc = self.textCursor()
        if tc.hasSelection():
            tc.removeSelectedText()
            # TOOD: group substituion
            tc.insertText(exp)

    def changedOnDisk(self):
        """We've been notified that the disk copy is changed. Set modified and notify the user they may want to reload."""
        # Its possible that this event was triggered by a save and the disk contents aren't actually different than what is displayed
        # this got more reliable when I took the read out of the condition?
        a = open(self.filename).read()
        b = self.toPlainText()
        if a != b:
            self.document().setModified(True)
            self.message = "File changed on disk!"
            self.updateTitleBar()

    def keyPressEvent(self, event):
        #Logger.debug(f"Editor keyPressEvent: {event.key()}, {int(event.modifiers())}")
        if event.key() in (Qt.Key_Shift, Qt.Key_Control, Qt.Key_Alt, Qt.Key_Meta):
            # handle lone modifiers
            return
        shortcut = QKeySequence(event.key() + int(event.modifiers())).toString()
        if ActionDispatcher().dispatchKey(shortcut):
            self.updateTitleBar()
            return
        QTextEdit.keyPressEvent(self, event)
        self.updateTitleBar()

    def isUnsaved(self):
        return self.document().isModified()

    def label(self):
        unsaved = self.isUnsaved() and "*" or ""
        return f"{self.filename}{unsaved} {self.message}"

    def updateTitleBar(self):
        # TODO: show VCS tracked & changed status
        unsaved = self.isUnsaved() and "*" or ""
        MainGui().updateTitleBar(f"{self.filename}{unsaved} {self.message}")

    def saveState(self):
        # must return an object that can be yaml serialized
        return {
            "filename": self.filename,
            "position": self.textCursor().position(),
            "scrollpos": self.verticalScrollBar().value(),
            "modified": self.isUnsaved(),
        }

    def loadState(self, state):
        # loads the state from a yaml record
        # not that filename has already been set by a call to open()
        # and subsequently load()
        tc = self.textCursor()
        tc.setPosition(state["position"])
        self.setTextCursor(tc)
        self.verticalScrollBar().setValue(state["scrollpos"])

    def autocomplete(self):
        tc = self.textCursor()
        # jedi is not 0 based
        line = tc.blockNumber() + 1
        column = tc.positionInBlock()
        generateSuggestions = Lightsaber().autocomplete(
            line, column, self.toPlainText(), self.filename
        )
        MainGui().autocompletePicker.popup(
            generateSuggestions, self.insertAutocomplete
        )

    def insertAutocomplete(self, result):
        tc = self.textCursor()
        tc.movePosition(QTextCursor.StartOfLine)
        tc.movePosition(QTextCursor.EndOfLine, QTextCursor.KeepAnchor)
        tc.deleteChar()
        tc.insertText(result)

    def underCursor(self):
        """Returns the word under the cursor"""
        tc = self.textCursor()
        if not tc.hasSelection():
            tc.movePosition(QTextCursor.EndOfWord)
            tc.movePosition(QTextCursor.StartOfWord, QTextCursor.KeepAnchor)
        return tc.selectedText()

    def indent(self):
        tc = self.textCursor()
        if not tc.hasSelection():
            self.insertPlainText(self.tab)
        else:
            doc = self.document()
            endBlock = doc.findBlock(tc.selectionEnd())
            tc.setPosition(tc.selectionStart())
            tc.movePosition(QTextCursor.StartOfLine)
            while True:
                tc.insertText(self.tab)
                if doc.findBlock(tc.position()) == endBlock:
                    break
                tc.movePosition(QTextCursor.Down)
                tc.movePosition(QTextCursor.StartOfLine)

    def unindent(self):
        tc = self.textCursor()
        tc.beginEditBlock()
        if not tc.hasSelection():
            tc.movePosition(QTextCursor.StartOfLine)
            for f in range(len(self.tab)):
                tc.movePosition(QTextCursor.NextCharacter, QTextCursor.KeepAnchor)
            if tc.selectedText() == self.tab:
                tc.deleteChar()
        else:
            doc = self.document()
            endBlock = doc.findBlock(tc.selectionEnd())
            tc.setPosition(tc.selectionStart())
            while True:
                tc.movePosition(QTextCursor.StartOfLine)
                pos = tc.position()
                for f in range(len(self.tab)):
                    tc.movePosition(QTextCursor.NextCharacter, QTextCursor.KeepAnchor)
                if tc.selectedText() == self.tab:
                    tc.removeSelectedText()
                else:
                    tc.setPosition(pos)
                if tc.position() >= endBlock.position():
                    break
                tc.movePosition(QTextCursor.Down)
        tc.endEditBlock()

    def insert_newline(self):
        tc = self.textCursor()
        pos = tc.position()
        tc.movePosition(QTextCursor.StartOfLine, QTextCursor.KeepAnchor)
        line = tc.selectedText()
        indent = 0
        while line.startswith(self.tab):
            indent += 1
            line = line[len(self.tab) :]
        if re.match(self.autoIndentRE, line):
            indent += 1
        tc.setPosition(pos)
        tc.insertText(self.newline + self.tab * indent)
        self.setTextCursor(tc)

    def moveCursor(self, moveop, select=False, retain_original=False, n=1):
        """Move the cursor based on QTextCursor.MoveOperation (moveop)"""
        if not select:
            mode = QTextCursor.MoveAnchor
        else:
            mode = QTextCursor.KeepAnchor
        tc = self.textCursor()
        for i in range(n):
            tc.movePosition(moveop, mode)
        if not retain_original:
            self.setTextCursor(tc)

    def copyLine(self):
        tc = self.textCursor()
        pos = tc.position()
        tc.movePosition(QTextCursor.StartOfBlock)
        tc.movePosition(QTextCursor.EndOfBlock, QTextCursor.KeepAnchor)
        self.setTextCursor(tc)
        self.copy()
        tc.setPosition(pos)
        self.setTextCursor(tc)


class Editor(QWidget):
    def __init__(self, ctrl):
        QWidget.__init__(self)
        self.ctrl = ctrl
        self.toolKey = "editor"
        self.defaultLocation = "main"
        self.label = lambda: self.main.label()
        self.lo = QHBoxLayout(self)
        self.lo.setContentsMargins(1, 1, 1, 1)
        self.lo.setSpacing(0)
        self.margin = Margin()
        self.main = EditorArea(ctrl)
        self.margin.wheelHandler = self.main.wheelEvent
        self.setFocusProxy(self.main)
        self.marks = Marks()
        self.overview = Overview()
        self.overview.wheelHandler = self.main.wheelEvent
        self.lo.addWidget(self.margin)
        self.lo.addWidget(self.main)
        self.lo.addWidget(self.marks)
        self.lo.addWidget(self.overview)

        # link the main text scrollbar to the line number scrollbar
        self.main.verticalScrollBar().valueChanged.connect(self._updateScroll)
        self.main.document().blockCountChanged.connect(self._updateNums)
        # self.margin.verticalScrollBar().valueChanged.connect(self._updateScroll)
        # self.overview.verticalScrollBar().valueChanged.connect(self._updateScroll)

        Watcher().addReceiver(self.fileChanged)

        # Public Interface (lazy version)
        self.filename = lambda: self.main.filename
        self.new = self.main.new
        self.save = self.main.save
        self.saveAs = self.main.saveAs
        self.isUnsaved = self.main.isUnsaved
        self.loadState = self.main.loadState
        self.saveState = self.main.saveState
        self.copy = self.main.copy
        self.cut = self.main.cut
        self.paste = self.main.paste
        self.search = self.main.search
        self.replace = self.main.replace
        self.autocomplete = self.main.autocomplete
        self.underCursor = self.main.underCursor
        # Define actions
        A = ActionEnum()
        A.indent.addHandler(self.main.indent, InFocus(self))
        A.unindent.addHandler(self.main.unindent, InFocus(self))
        A.insert_newline.addHandler(self.main.insert_newline, InFocus(self))
        A.delete_left.addHandler(
            lambda: self.main.textCursor().deletePreviousChar(), InFocus(self)
        )
        A.delete_right.addHandler(
            lambda: self.main.textCursor().deleteChar(), InFocus(self)
        )
        A.cursor_up.addHandler(
            lambda: self.main.moveCursor(QTextCursor.Up), InFocus(self)
        )
        A.cursor_down.addHandler(
            lambda: self.main.moveCursor(QTextCursor.Down), InFocus(self)
        )
        A.cursor_left.addHandler(
            lambda: self.main.moveCursor(QTextCursor.Left), InFocus(self)
        )
        A.cursor_right.addHandler(
            lambda: self.main.moveCursor(QTextCursor.Right), InFocus(self)
        )
        A.word_left.addHandler(
            lambda: self.main.moveCursor(QTextCursor.WordLeft), InFocus(self)
        )
        A.word_right.addHandler(
            lambda: self.main.moveCursor(QTextCursor.WordRight), InFocus(self)
        )
        # TODO: figure out page size?
        A.page_up.addHandler(
            lambda: self.main.moveCursor(QTextCursor.Up, n=50), InFocus(self)
        )
        A.page_down.addHandler(
            lambda: self.main.moveCursor(QTextCursor.Down, n=50), InFocus(self)
        )
        A.line_home.addHandler(
            lambda: self.main.moveCursor(QTextCursor.StartOfLine), InFocus(self)
        )
        A.line_end.addHandler(
            lambda: self.main.moveCursor(QTextCursor.EndOfLine), InFocus(self)
        )
        A.file_home.addHandler(
            lambda: self.main.moveCursor(QTextCursor.Start), InFocus(self)
        )
        A.file_end.addHandler(
            lambda: self.main.moveCursor(QTextCursor.End), InFocus(self)
        )
        A.copy_line.addHandler(self.main.copyLine, InFocus(self))

    ## Public interface (better version)
    def load(self, filename):
        self.main.load(filename)
        self.overview.load(filename)

    ## Private methods
    def _updateScroll(self, val):
        """Scrolling on the main area must change the scroll position
        of the margin, the marks and the overview"""
        # TODO: _updateNums here is kind of lazy, should trigger on QTextDocumentLayout
        # changes, somehow.
        self._updateNums()
        self.margin.verticalScrollBar().setValue(val)
        startBlock = self.main.cursorForPosition(QPoint(0, 0)).block().blockNumber()
        h = self.main.size().height()
        endBlock = self.main.cursorForPosition(QPoint(0, h)).block().blockNumber()
        self.overview.updateCursor(startBlock, endBlock)
        # assumes minimums are 0
        scaledScrollPos = (
            val
            / self.main.verticalScrollBar().maximum()
            * self.overview.verticalScrollBar().maximum()
        )
        self.overview.verticalScrollBar().setValue(scaledScrollPos)

    def _updateNums(self):
        # update the line numbers when the main document block count changes
        # OR when it scrolls b/c QTextDocument does lazy layout
        doc = self.main.document()
        count = doc.blockCount()
        width = int(math.log(count, 10)) + 1
        self.margin.setFixedWidth(14 * width + 8)
        linenums = []
        for n in range(count):
            linenums.append(str(n).rjust(width))
            tb = doc.findBlockByNumber(n)
            lines = tb.layout().lineCount()
            if lines > 1:
                linenums.extend([" " * width for f in range(lines - 1)])
        self.margin.setPlainText("\n".join(linenums))

    def fileChanged(self, filename):
        if filename == self.filename():
            self.main.changedOnDisk()
