import time
from enum import Enum

from PySide2.QtCore import Signal, QThread, QObject, QProcess, QTimer

from source.logger import Logger


class GeneratorThread(QThread):
    """A child thread runs until the given generator is empty"""

    BUFFER_TIME = 0.1  # seconds

    def __init__(self, generator, args, kwargs, resultsHandler, finishedHandler):
        QThread.__init__(self)
        self.generator = generator
        self.args = args
        self.kwargs = kwargs
        self.resultsHandler = resultsHandler
        self.finishedHandler = finishedHandler

    def run(self):
        timer = time.time()
        buf = []
        try:
            for entry in self.generator(*self.args, **self.kwargs):
                buf.append(entry + "\n")
                now = time.time()
                if now - timer > self.BUFFER_TIME:
                    self.resultsHandler("".join(buf))
                    if self.isInterruptionRequested():
                        return
                    timer = now
                    buf = []
        except Exception as e:
            Logger.error(f"GeneratorThread raised: {repr(e)}")
            return
        self.resultsHandler("".join(buf))
        self.finishedHandler()


class GeneratorOutputHandler(QObject):
    resultsReady = Signal(str)
    finished = Signal()

    def __init__(self, generator, args=(), kwargs={}):
        QObject.__init__(self)
        self._createChild(generator, args, kwargs)

    def _createChild(self, generator, args, kwargs):
        self.child = GeneratorThread(generator, args, kwargs, self._emitResults, self._emitFinished)

    def _emitResults(self, results):
        self.resultsReady.emit(results)

    def _emitFinished(self):
        self.finished.emit()

    def start(self):
        self.child.start()

    def changeGenerator(self, generator):
        # change the generator
        self.stopThread()
        self._createChild(generator)

    def stopThread(self):
        # stop the child thread
        self.child.requestInterruption()
        self.child.wait()


class SubprocessHandler(QObject):
    BUFFER_TIME = 0.1  # seconds
    resultsReady = Signal(str)
    finished = Signal()

    def __init__(self):
        QObject.__init__(self)
        self.timer = QTimer(self)
        self.timer.setInterval(self.BUFFER_TIME * 1000)
        self.timer.timeout.connect(self.readBuffers)
        self.child = QProcess(self)
        # TODO: implement I/O
        self.child.setProcessChannelMode(QProcess.MergedChannels)
        self.child.finished.connect(self.processFinished)
        self.child.errorOccurred.connect(self.processFinished)

    def _read(self):
        result = self.child.readAll().data().decode("utf-8", "replace")
        return result

    def readBuffers(self):
        result = self._read()
        if result:
            self.resultsReady.emit(result)

    def processFinished(self):
        self.readBuffers()
        self.child.close()
        self.timer.stop()
        self.finished.emit()

    def start(self, args):
        self.args = args
        if self.child.state() == QProcess.NotRunning:
            self.child.start(self.args[0], self.args[1:])
            self.timer.start()
        else:
            # TODO: implement stack
            pass

    def stopThread(self):
        self.child.close()
        self.child.waitForFinished()
