class NoOverwriteException(Exception):
    pass


""" I initially attempted this pattern with a closure, unfortunately closures
can't seem to modify the closed states, probably a good thing.
Using a class is more explicit anyway.
Why not just use MySingleton = MyClass() at the module level? b/c this delays calling
MyClass.__init__ until the first time you need to call Singleton(MyClass)() ie not
on import.
"""


class Singleton:
    """NOT A META-CLASS. Wrap the class you wish to use as a singleton at the module
    level like this: MyModuleSingleton = Singleton(MyClass)
    Other scripts that import MyModuleSingleton can use MyModuleSingleton()
    to always refer to the single instance of the MyClass.
    """

    def __init__(self, cls):
        self.cls = cls
        self.ref = None

    def __call__(self):
        if not self.ref:
            self.ref = self.cls()
        return self.ref


class NoOverwrite:
    """This metaclass prevents object members from being overwritten once set to
    something other than an empty tuple."""

    def __setattr__(self, k, v):
        if hasattr(self, k) and getattr(self, k) != ():
            raise NoOverwriteException(
                f"You cannot overwrite member: {k} of class {self.__class__}."
            )
        super().__setattr__(k, v)


class AutoNamedMembers:
    """A metaclass, where members of the child class (usually initialized to
    a dummy value like () or None) will be converted to object members
    containing instances of __member_cls__.
    """

    __member_cls__ = None

    def __init__(self):
        for k in self.__class__.__dict__.keys():
            if not k.startswith("_") and not k in self.__dict__:
                setattr(self, k, None)

    def __setattr__(self, k, v):
        if v == () or v is None:
            v = self.__member_cls__(k)
        super().__setattr__(k, v)
