from source.basecls import Singleton, NoOverwrite, AutoNamedMembers
from source.logger import Logger


class _ActionDispatcher:
    """Takes actions, writes them to the log and calls their handlers"""

    def __init__(self):
        self.ctrl = None
        self._log_function = None

    def setup_actions(self, ctrl, actionEnum):
        self.ctrl = ctrl
        # makes connections b/t individual actions and this.
        self.actionEnum = actionEnum
        for action in self.actionEnum:
            action.dispatcher = self

    def connect_log(self, log):
        self._log_function = log

    def log(self, msg):
        if self._log_function:
            self._log_function(msg)

    def cli(self, arg_string):
        """Execute the action with the given keyword arguments. Normally
        how actions are dispatched from the command line interface (CLI)
        """
        try:
            o = compile(arg_string, "<CLI>", "eval")
            exec(o, globals(), ActionEnum().__dict__)
        except NameError as ne:
            # terminal will catch this
            Logger.info(repr(ne))
            raise
        except Exception as e:
            Logger.info(repr(e))
            self.log(repr(e))

        return True

    def dispatchKey(self, key):
        """ takes a QKeySequence (ie from a keyPressEvent) and checks it against
        all the registered actions. This allows actions to use keys like Tab or Return
        even when they are normally used by QTextEdit (for example).
        dispatchKey must be able to signal the calling keyPressEvent whether the key
        was handled by an action or not. False indicates it was not and the calling
        keyPressEvent should handle it.
        """
        Logger.debug(f"Dispatch key {key}")
        for action in self.actionEnum:
            if action.shortcut == key:
                action()
                return True
        return False

    def validate_shortcut(self, action, shortcut):
        for a in self.actionEnum:
            if a is not action and a.shortcut == shortcut:
                raise Exception(
                    f"Duplicate shortcut {shortcut} on Action.{action.name}"
                )


# convenient functions (closures) for addHandler's when argument
def Always(obj=None):
    return lambda ctrl: True


def InFocus(obj):
    def _InFocus(ctrl):
        return obj.hasFocus()
    return _InFocus

    

class Action:
    def __init__(self, name):
        self.name = name
        # TODO: verify shortcuts aren't used twice
        self.shortcut = None
        # TODO: include an argument parser for commands
        self.parser = None
        self.menuText = None
        self.handlers = []
        # dispatcher is set by ActionDispatcher.setup_actions(ActionEnum)
        self.dispatcher = None

    def __call__(self, *args, **kwargs):
        Logger.debug(f"Action ({self.name}) Called")
        for when, handler in self.handlers:
            if when(self.dispatcher.ctrl):
                handler(*args, **kwargs)

    def C(self, shortcut=None, menuText=None, **kwargs):
        self.shortcut = shortcut
        if shortcut:
            self.dispatcher.validate_shortcut(self, shortcut)
        self.menuText = menuText

    def addHandler(self, handler, when=None):
        """ Add a bound method as a handler for the action. 
            when must be a callable that accepts self.ctrl.
            The default action is to check if the method's parent object
            has the UI focus before calling the method.
        """
        # TODO: first handler sets the arg spec
        # TODO: further handlers arg spec must match
        # TODO: CLI dispatch includes an arg validate step
        # which could be as simple as try/except
        if not when:
            #when = InFocus(parent)
            when = Always() # TODO: siwtch default to InFocus
        assert callable(when)
        assert callable(handler)
        self.handlers.append((when, handler))


# It's pretty weird to me that this works: NoOverwrite.__setattr__ is called before AutoNameMembers.__setattr__
class _ActionEnum(NoOverwrite, AutoNamedMembers):
    __member_cls__ = Action
    new_file = ()
    open_file = ()
    save_file = ()
    save_as_file = ()
    close_file = ()
    remove_file = ()
    create_project = ()
    open_project = ()
    quit = ()
    open_browser = ()
    get_web_page = ()
    get_latest_page = ()
    copy_selection = ()
    cut_selection = ()
    paste_from_clipboard = ()
    show_find_dock = ()
    show_replace_dock = ()
    focus_on_main_tool = ()
    focus_next_tool = ()
    focus_previous_tool = ()
    show_pyhelp_dock = ()
    show_open_files_dock = ()
    show_project_dock = ()
    show_terminal_dock = ()
    show_file_manager_dock = ()
    toggle_between_focus_dock = ()
    grow_dock = ()
    shrink_dock = ()
    open_settings_file = ()
    reload_settings_file = ()
    suggest_autocomplete = ()
    print_files = ()
    copy_files = ()
    find_files = ()
    search_in_files = ()
    list_files = ()
    md5sum_file = ()
    move_files = ()
    remove_files = ()
    create_files = ()
    ztar = ()
    compress_files = ()
    decompress_files = ()
    delete_left = ()
    delete_right = ()
    cursor_up = ()
    cursor_down = ()
    cursor_left = ()
    cursor_right = ()
    word_left = ()
    word_right = ()
    page_up = ()
    page_down = ()
    line_home = ()
    line_end = ()
    file_home = ()
    file_end = ()
    copy_line = ()
    cut_line = ()
    indent = ()
    unindent = ()
    insert_newline = ()
    select_all = ()
    undo = ()
    redo = ()

    def __init__(self):
        # Due to AutoNamedMembers, members above will be instances of Action
        AutoNamedMembers.__init__(self)

    def __iter__(self):
        return iter([member for key, member in self.__dict__.items() if key[0] != "_"])

    def __contains__(self, key):
        return key in self.__dict__


# a singleton, you could call ActionDispatcher() as often as you want
# but not a module level object so that __init__ is run late, not on import
ActionDispatcher = Singleton(_ActionDispatcher)
ActionEnum = Singleton(_ActionEnum)
