import os
import yaml
import logging
import argparse
from dataclasses import asdict

from PySide2.QtWidgets import (
    QApplication,
    QFileDialog,
    QInputDialog,
    QMessageBox,
    QLineEdit,
)
from PySide2.QtCore import Qt, QByteArray, QFileSystemWatcher

from source.logger import Logger
from source.gui import MainGui
import source.projects as projects
from source.action import ActionEnum, ActionDispatcher
from source.toolregistry import ToolRegistry, ToolKeys
from source.toolfactory import ToolFactory
from source.resources import Settings
from source.theforce import Lightsaber

# BACKGROUND PROCESSES
from source.watcher import Watcher



class Main:
    def __init__(self):
        parser = argparse.ArgumentParser(
            prog="shokunin", description="A craftsman's development environment."
        )
        parser.add_argument("-p", "--project", type=str, help="project to load")
        parser.add_argument("-d", "--debug", action="store_true", help="Enable debug log messages.")
        parser.add_argument("files", nargs="*", type=str, help="files to load")
        self.args = parser.parse_args()
        if self.args.debug:
            Logger.info("Setting logger to DEBUG")
            Logger.setLevel(level=logging.DEBUG)
        # this is the QMainWindow
        self.gui = None
        # The project object, if one is loaded
        self.project = None
        # the dispatcher handles all user actions (someday)
        ActionDispatcher().setup_actions(self, ActionEnum())
        # registry of all open tools
        ToolRegistry().setup()
        # provides language specific features (autoformat, autocomplete, etc)
        Lightsaber().setup()
        # load Settings
        Settings().load_settings()

        # assign action handlers
        ActionEnum().open_project.addHandler(self.openProject)
        ActionEnum().create_project.addHandler(self.createProject)

        self.childThreads = []
        self.app = QApplication([])
        Watcher().setup()
        self.gui = MainGui()

        # connect the gui to stuff
        Lightsaber().connect_gui(self.gui)
        ToolRegistry().connect_gui(self, self.gui, ToolFactory)
        ActionDispatcher().connect_log(self.gui.log)

        self.gui.spawnDefaultTools()
        # WORK AROUND TO MAKE showMaximized work reliably on Windows
        self.gui.resize(800, 600)
        self.gui.showMaximized()

        if self.args.project:
            self.openProject(self.args.project)
        if self.args.files:
            for f in self.args.files:
                self.gui.open(f)

        # Main thread
        self.app.exec_()
        
        # TODO: trigger this on an exception?
        # stop any child thread here
        for t in self.childThreads:
            t.stopThread()


    def createProject(self):
        result = QInputDialog().getText(
            self.gui, "Enter a project result", "name:", QLineEdit.Normal, ""
        )
        if not result:
            return
        name = result[0]
        result = QFileDialog.getExistingDirectory(
            self.gui,
            caption="Select the project directory",
            options=QFileDialog.DontUseNativeDialog | QFileDialog.ShowDirsOnly,
        )
        if not result:
            return
        path = result
        if not os.path.exists(path):
            return
        project = projects.loadProject(path)
        if project:
            result = QMessageBox.question(
                self.gui,
                "",
                "Project file already exists in {}. Do you want to overwrite?".format(
                    projectfile
                ),
            )
            if result != QMessageBox.Yes:
                return
        self.project = projects.createProject(name, path)
        os.chdir(self.project.path)
        self.lsp.change_project(self.project.path)
        self.gui.projectStatus.updateProject(self.project)
        # TODO: project templates

    def openProject(self, projectPath=None):
        if not projectPath:
            result = QFileDialog.getExistingDirectory(
                self.gui, options=QFileDialog.DontUseNativeDialog, dir=os.getcwd()
            )
            if not result:
                return
            projectPath = result
        self.project = projects.loadProject(projectPath)
        os.chdir(self.project.path)
        self.lsp.change_project(self.project.path)

        self.gui.projectStatus.updateProject(self.project)
        for record in self.project.savedStates:
            if not record["filename"]:
                self.gui.new()
            else:
                self.gui.open(record["filename"])
            self.gui.currentTool.loadState(record)
        self.gui.restoreGeometry(
            QByteArray.fromBase64(QByteArray(self.project.geometry.encode()))
        )
        self.gui.restoreState(
            QByteArray.fromBase64(QByteArray(self.project.docks.encode()))
        )


if __name__ == "__main__":
    Main()
