""" 
shokunin - 職人
mastery of craft
art, function, product, process
each part of the whole
to better our world
"""

import re
import io
import os
import sys
import shlex
import yaml
import toml
import argparse
import subprocess
from collections import defaultdict
from dataclasses import dataclass, asdict
from typing import *

from PySide2.QtWidgets import (
    QApplication,
    QWidget,
    QMainWindow,
    QTextEdit,
    QLineEdit,
    QListWidget,
    QListWidgetItem,
    QPushButton,
    QRadioButton,
    QCheckBox,
    QStatusBar,
    QMenu,
    QMenuBar,
    QAction,
    QToolTip,
    QVBoxLayout,
    QHBoxLayout,
    QStackedLayout,
    QFormLayout,
    QLabel,
    QFileDialog,
    QInputDialog,
    QMessageBox,
    QDockWidget,
    QAbstractScrollArea,
)

from PySide2.QtGui import (
    QSyntaxHighlighter,
    QTextDocument,
    QTextCharFormat,
    QBrush,
    QColor,
    QTextBlockUserData,
    QTextCursor,
    QKeySequence,
    QFont,
    QIcon,
)
from PySide2.QtCore import Qt, QRegularExpression

from pygments.token import Token, string_to_tokentype
from pygments.lexers import PythonLexer


EXEDIR = (
    hasattr(sys, "_MEIPASS")
    and os.path.abspath(os.path.dirname(sys.executable))
    or os.path.abspath(os.path.dirname(__file__))
)


def RESOURCE(filename):
    # TODO: load from user folder first
    return os.path.join(EXEDIR, "data", filename)


SETTINGS = {"style": "default"}
try:
    SETTINGS.update(yaml.safe_load(open(RESOURCE("settings.yaml")).read()))
except Exception as e:
    print(str(e))

# TODO: improve windows styling, menu colors, fileDialog headers, font
try:
    STYLESHEET = open(RESOURCE(SETTINGS["style"] + ".css")).read()
except Exception as e:
    print(str(e))
    STYLESHEET = """
QWidget {
    background-color: #111111;
    color: #EEEEEE;
    font-family: "DejaVu Sans Mono";
    font-size: 13pt;
    border: 1px solid black;
    padding: 0;
    margin: 0;
}
QTextEdit {
    background-color: #000000;
}
QMenuBar {
    font-weight: bold;
}
QListWidget {
    font-size: 11pt;
}
"""

# color, italic, bold
HILITESTYLES = {
    "Comment": (0x00AA00, True, False),
    "Literal.String.Doc": (0x00AA00, True, False),
    "Keyword": (0xAA00FF, False, False),
    "Name": (0xFFFFFF, False, False),
    "Name.Function": (0xAA5500, False, False),
    "Name.Class": (0xAAAA00, False, True),
    "Literal.String": (0x5555FF, False, False),
    "Literal": (0xAA0000, False, False),
    "Operator": (0xAA0000, False, False),
    "Text": (0xFFFFFF, False, False),
}
try:
    HILITESTYLES.update(yaml.safe_load(open(RESOURCE("hilite.yaml"))))
except Exception as e:
    print(str(e))


@dataclass
class Project:
    name: str
    path: str
    vcs: str
    savedStates: List[str]


class SearchField(QLineEdit):
    def __init__(self, dock):
        QLineEdit.__init__(self)
        self.dock = dock

    def focusOutEvent(self, event):
        QLineEdit.focusOutEvent(self, event)
        self.dock.focusOutEvent(event)


class ReplaceBox(QDockWidget):
    def __init__(self, parent, ctrl):
        QDockWidget.__init__(self, "Find & Replace", parent)
        self.setAllowedAreas(Qt.TopDockWidgetArea | Qt.BottomDockWidgetArea)
        self.ctrl = ctrl

        self.main = QWidget()
        self.findEdit = QLineEdit()
        self.replaceEdit = QLineEdit()
        self.replaceEdit.returnPressed.connect(self.replaceOne)
        self.findLabel = QLabel("Find:")
        self.replaceLabel = QLabel("Replace:")
        self.inLabel = QLabel("in")
        self.lineOpt = QRadioButton("&line", self.main)
        self.selectionOpt = QRadioButton("&selection", self.main)
        self.fileOpt = QRadioButton("&file", self.main)
        self.projectOpt = QRadioButton("&project", self.main)
        self.caseOpt = QCheckBox("Case &insensitive")
        # TODO: replace one button
        self.allBtn = QPushButton("Replace &All")
        self.allBtn.pressed.connect(self.replaceAll)

        self.lo = QHBoxLayout(self.main)
        self.lo.addWidget(self.findLabel)
        self.lo.addWidget(self.findEdit)
        self.lo.addWidget(self.replaceLabel)
        self.lo.addWidget(self.replaceEdit)
        self.lo.addWidget(self.inLabel)
        self.lo.addWidget(self.lineOpt)
        self.lo.addWidget(self.selectionOpt)
        self.lo.addWidget(self.fileOpt)
        self.lo.addWidget(self.projectOpt)
        self.lo.addWidget(self.caseOpt)
        self.lo.addWidget(self.allBtn)
        self.setWidget(self.main)
        self.hide()

    def show(self):
        QDockWidget.show(self)
        self.findEdit.setFocus()

    def setFocus(self):
        self.findEdit.setFocus()

    def hasFocus(self):
        return (
            self.findEdit.hasFocus()
            or self.replaceEdit.hasFocus()
            or self.lineOpt.hasFocus()
            or self.selectionOpt.hasFocus()
            or self.fileOpt.hasFocus()
            or self.projectOpt.hasFocus()
            or self.caseOpt.hasFocus()
            or self.allBtn.hasFocus()
        )

    def replaceAll(self):
        pass

    def replaceOne(self):
        # TODO: highlight/confirm
        needle = self.findEdit.text()
        if self.caseOpt.isChecked():
            self.ctrl.gui.currentTool.search(needle, repeat=True)
        else:
            self.ctrl.gui.currentTool.search(
                needle, repeat=True, flags=QTextDocument.FindCaseSensitively
            )
        self.ctrl.gui.currentTool.replace(self.replaceEdit.text())


class SearchBox(QDockWidget):
    def __init__(self, parent, ctrl):
        QDockWidget.__init__(self, "Find", parent)
        self.setAllowedAreas(Qt.TopDockWidgetArea | Qt.BottomDockWidgetArea)
        self.ctrl = ctrl
        self.main = QWidget()
        self.lo = QHBoxLayout(self.main)
        self.findEdit = SearchField(self)
        self.findEdit.returnPressed.connect(self.nextResult)
        self.findEdit.textChanged.connect(self.enterTerm)
        self.caseOpt = QCheckBox("Case &insensitive")
        self.lo.addWidget(self.findEdit)
        self.lo.addWidget(self.caseOpt)
        self.setWidget(self.main)
        self.setFocus = self.findEdit.setFocus
        self.hasFocus = self.findEdit.hasFocus

    def show(self):
        QDockWidget.show(self)
        self.findEdit.setFocus()

    def focusOutEvent(self, event):
        if self.findEdit.hasFocus() or self.caseOpt.hasFocus():
            return
        self.closeDock()

    def closeDock(self):
        self.ctrl.gui.removeDockWidget(self)

    def nextResult(self):
        self.enterTerm(repeat=False)

    def enterTerm(self, repeat=True):
        needle = self.findEdit.text()
        if self.caseOpt.isChecked():
            self.ctrl.gui.currentTool.search(needle, repeat=repeat)
        else:
            self.ctrl.gui.currentTool.search(
                needle, repeat=repeat, flags=QTextDocument.FindCaseSensitively
            )


def makeFormat(foreground, italic=False, bold=False, underline=False, background=None):
    """ create a QTextCharFormat based on some common attributes """
    tcFormat = QTextCharFormat()
    tcFormat.setForeground(QBrush(QColor(foreground)))
    if background:
        tcFormat.setBackground(QBrush(QColor(background)))
    if italic:
        tcFormat.setFontItalic(True)
    if bold:
        tcFormat.setFontWeight(100)
    return tcFormat


class UserData(QTextBlockUserData):
    """ Stores the highlighter state of the current textblock """

    def __init__(self, stack):
        QTextBlockUserData.__init__(self)
        self.stack = stack


class Highlighter(QSyntaxHighlighter):
    """ Pygments based highlighter class """

    def __init__(self, doc):
        QSyntaxHighlighter.__init__(self, doc)
        self.lex = PythonLexer()
        # WARNING: a typo on Token.XX not throw an error
        self.defaultStyle = makeFormat(*HILITESTYLES["Text"])
        self.styles = {}
        for tokenString, props in HILITESTYLES.items():
            self.styles[string_to_tokentype(tokenString)] = makeFormat(*props)

    def styleMatcher(self, token):
        match = None
        for tokenType in self.styles:
            if token in tokenType and (not match or len(tokenType) > len(match)):
                match = tokenType
        return match

    def highlightBlock(self, text):
        block = self.currentBlock()
        userdata = block.previous().userData()
        if not userdata or not userdata.stack:
            stack = ("root",)
        else:
            stack = userdata.stack
        for r in self.lex.get_tokens_unprocessed(text, stack):
            i, tt, v, stack = r
            match = self.styleMatcher(tt)
            if match:
                self.setFormat(i, len(v), self.styles[match])
            else:
                # This is here to fix the weird missing lines bug
                # that I noticed on some nested tuples but had
                # trouble finding a minimal example that reproduced
                # it's not related to block.setVisible
                # and setting the textCharFormat fixed it
                # it's unclear to me what happening
                self.setFormat(i, len(v), self.defaultStyle)
        self.setCurrentBlockUserData(UserData(stack))


class Editor(QTextEdit):
    def __init__(self, ctrl):
        QTextEdit.__init__(self)
        self.ctrl = ctrl
        self.toolKey = "edit"
        self.mode = "insert"
        self.autoIndentRE = ".+:$"
        self.tab = "    "
        self.newline = "\n"
        # TODO: add line numbers
        # TODO: add cursor position to status bar

        self.new()

    def setFilename(self, filename):
        self.filename = filename
        self.document().setModified(False)
        self.updateTitleBar()

    def load(self, filename):
        self.setPlainText(open(filename).read())
        if filename.endswith(".py"):
            self.hl = Highlighter(self.document())
        self.setFilename(filename)

    def save(self):
        if self.filename:
            open(self.filename, "w").write(self.toPlainText())
            if self.filename.endswith(".py"):
                # TODO: improve speed of black (--fast didn't help much)
                result = subprocess.run(
                    shlex.split("python3 -m black --fast " + self.filename),
                    capture_output=True,
                )
                self.ctrl.gui.displayUserMessage(
                    result.stderr.decode("utf-8").splitlines()[0]
                )
            pos = self.textCursor().position()
            scrollPos = self.verticalScrollBar().value()
            self.load(self.filename)
            tc = self.textCursor()
            tc.setPosition(pos)
            self.setTextCursor(tc)
            self.verticalScrollBar().setValue(scrollPos)

    def saveAs(self, filename):
        self.setFilename(filename)
        self.save()

    def new(self):
        self.setFilename(None)
        self.setPlainText("")

    def search(self, exp, direction=None, repeat=False, flags=None):
        if repeat:
            tc = self.textCursor()
            tc.setPosition(tc.anchor())
            # if you don't do this it searches from the end of the current selection
            # messing up search as you type
            self.setTextCursor(tc)
        regExp = QRegularExpression(exp)
        secondPos = self.document().begin().position()
        if direction == "back":
            flags |= QTextDocument.FindBackward
            secondPos = self.document().end().position()
        if flags:
            found = self.find(QRegularExpression(exp), flags)
        else:
            found = self.find(QRegularExpression(exp))
        if not found:
            # try again with wrapping
            if flags:
                tc = self.document().find(regExp, secondPos, flags)
            else:
                tc = self.document().find(regExp, secondPos)
            if tc.hasSelection():
                self.setTextCursor(tc)
                found = True
        return found

    def replace(self, exp):
        """ replaces the current selection with the exp, performing group substitution """
        tc = self.textCursor()
        if tc.hasSelection():
            tc.removeSelectedText()
            # TOOD: group substituion
            tc.insertText(exp)

    def keyPressEvent(self, event):
        key = event.key()
        if key == Qt.Key_Tab:
            tc = self.textCursor()
            if not tc.hasSelection():
                self.insertPlainText(self.tab)
            else:
                doc = self.document()
                endBlock = doc.findBlock(tc.selectionEnd())
                tc.setPosition(tc.selectionStart())
                tc.movePosition(QTextCursor.StartOfLine)
                while True:
                    tc.insertText(self.tab)
                    if doc.findBlock(tc.position()) == endBlock:
                        break
                    tc.movePosition(QTextCursor.Down)
                    tc.movePosition(QTextCursor.StartOfLine)
        elif key == Qt.Key_Backtab:
            # TODO: single undo operation
            tc = self.textCursor()
            if not tc.hasSelection():
                tc.movePosition(QTextCursor.StartOfLine)
                for f in range(len(self.tab)):
                    tc.movePosition(QTextCursor.NextCharacter, QTextCursor.KeepAnchor)
                if tc.selectedText() == self.tab:
                    tc.deleteChar()
            else:
                doc = self.document()
                endBlock = doc.findBlock(tc.selectionEnd())
                tc.setPosition(tc.selectionStart())
                while True:
                    tc.movePosition(QTextCursor.StartOfLine)
                    for f in range(len(self.tab)):
                        tc.movePosition(
                            QTextCursor.NextCharacter, QTextCursor.KeepAnchor
                        )
                    if tc.selectedText() == self.tab:
                        tc.removeSelectedText()
                    if doc.findBlock(tc.position()) == endBlock:
                        break
                    tc.movePosition(QTextCursor.Down)
        elif key == Qt.Key_Return:
            tc = self.textCursor()
            pos = tc.position()
            tc.movePosition(QTextCursor.StartOfLine, QTextCursor.KeepAnchor)
            line = tc.selectedText()
            indent = 0
            while line.startswith(self.tab):
                indent += 1
                line = line[len(self.tab) :]
            if re.match(self.autoIndentRE, line):
                indent += 1
            tc.setPosition(pos)
            tc.insertText(self.newline + self.tab * indent)
            self.setTextCursor(tc)
        else:
            QTextEdit.keyPressEvent(self, event)
        self.updateTitleBar()

    def isUnsaved(self):
        return self.document().isModified()

    def updateTitleBar(self):
        # TODO: show VCS tracked & changed status
        if self.isUnsaved():
            self.ctrl.gui.updateTitleBar("{}*".format(self.filename))
        else:
            self.ctrl.gui.updateTitleBar("{}".format(self.filename))

    def saveState(self):
        # must return an object that can be yaml serialized
        return {
            "filename": self.filename,
            "position": self.textCursor().position(),
            "scrollpos": self.verticalScrollBar().value(),
            "modified": self.isUnsaved(),
        }

    def loadState(self, state):
        # loads the state from a yaml record
        # not that filename has already been set by a call to open()
        # and subsequently load()
        tc = self.textCursor()
        tc.setPosition(state["position"])
        self.setTextCursor(tc)
        self.verticalScrollBar().setValue(state["scrollpos"])


class ActionBar(QMenuBar):
    def __init__(self, actions):
        QMenuBar.__init__(self)
        self.menus = {}
        for menuText, items in actions:
            menuObj = QMenu(menuText)
            for text, handler, shortcut in items:
                if text == "-":
                    menuObj.addSeparator()
                    continue
                menuObj.addAction(text, handler, shortcut)
            self.addMenu(menuObj)
            key = menuText.lower().replace("&", "")
            self.menus[menuText] = menuObj


class PythonHelper(QDockWidget):
    AUTOLIBS = (
        "",
        "PySide2.QtWidgets.",
        "PySide2.QtGui.",
        "PySide2.QtCore.",
        "os.",
        "sys.",
    )

    def __init__(self, title, parent, ctrl):
        QDockWidget.__init__(self, title, parent)
        self.setAllowedAreas(Qt.LeftDockWidgetArea | Qt.RightDockWidgetArea)
        self.ctrl = ctrl
        main = QWidget(self)
        self.content = QTextEdit()
        self.content.setReadOnly(True)
        self.content.setFocusPolicy(Qt.NoFocus)
        self.cli = QLineEdit()
        lo = QVBoxLayout(main)
        lo.addWidget(self.content)
        lo.addWidget(self.cli)
        self.setWidget(main)
        # TODO: keyboard navigation of the help pane
        self.cli.returnPressed.connect(self.resolve)

    def hasFocus(self):
        return self.cli.hasFocus()

    def setFocus(self):
        return self.cli.setFocus()

    def show(self):
        QDockWidget.show(self)
        self.cli.setFocus()

    def resolve(self):
        # Have to be sneaky here because help() writes to stdout
        for prefix in PythonHelper.AUTOLIBS:
            sneaky_io = io.StringIO()
            sys.stdout = sneaky_io
            val = prefix + self.cli.text()
            try:
                help(val)
            except:
                sys.stdout = sys.__stdout__
                raise
            finally:
                sys.stdout = sys.__stdout__
            result = sneaky_io.getvalue()
            if not result.startswith("No Python"):
                self.content.setText(result)
                break


class OpenFileItem(QListWidgetItem):
    def __init__(self, text, tool):
        QListWidgetItem.__init__(self, text)
        self.filename = text
        self.tool = tool


class OpenFileList(QDockWidget):
    def __init__(self, title, parent, ctrl):
        QDockWidget.__init__(self, title, parent)
        self.setAllowedAreas(Qt.LeftDockWidgetArea | Qt.RightDockWidgetArea)
        self.ctrl = ctrl
        self.main = QListWidget(self)
        self.setWidget(self.main)
        self.main.itemActivated.connect(self.change)

    def show(self):
        QDockWidget.show(self)
        self.main.setFocus()
        windowWidth = self.ctrl.gui.size().width()
        # TODO: factor is configurable
        proposedWidth = int(0.2 * windowWidth)
        if proposedWidth < 200:
            proposedWidth = 200
        self.setMinimumWidth(proposedWidth)

    def setFocus(self):
        self.main.setFocus()

    def hasFocus(self):
        return self.main.hasFocus()

    def addItem(self, tool):
        # TODO: scroll right if names are too long
        item = OpenFileItem(str(tool.filename), tool)
        self.main.addItem(item)
        sb = self.main.horizontalScrollBar()
        sb.setValue(sb.maximum())

    def removeItem(self, tool):
        for i in self.main.findItems(str(tool.filename), Qt.MatchExactly):
            if i.tool == tool:
                row = self.main.row(i)
                self.main.takeItem(row)

    def change(self, item):
        self.ctrl.gui.switchTool(item.tool)


class ProjectStatus(QDockWidget):
    def __init__(self, ctrl):
        QDockWidget.__init__(self, "Project Status", ctrl.gui)
        self.setAllowedAreas(Qt.LeftDockWidgetArea | Qt.RightDockWidgetArea)
        self.ctrl = ctrl

        self.fileList = QListWidget()
        self.fileList.itemActivated.connect(self.openFile)
        # TODO: make into file system tree view with subdirs folded by default
        # TODO: filter by vcs tracked status
        self.setWidget(self.fileList)
        self.setFocus = self.fileList.setFocus
        self.hasFocus = self.fileList.hasFocus

    def show(self):
        QDockWidget.show(self)
        self.fileList.setFocus()

    def openFile(self, item):
        self.ctrl.gui.currentTool.load(item.text())

    def setProject(self, project):
        self.setWindowTitle("{}: Project Status".format(project.name))
        self.project = project
        self.updateFileList()

    def updateFileList(self):
        path = self.project.path
        # TODO: filtering by .hgignore
        # TODO: custom filtering
        self.fileList.clear()
        # exclude_patterns = toml.loads(open("." + self.project.vcs + "ignore").read())
        # excludes = []
        self.fileList.addItems(os.listdir(path))


class VCSDock(QDockWidget):
    def __init__(self, ctrl):
        # TODO: use git|hg in title, tag, changed/commited status
        QDockWidget.__init__(self, "VCS Status", ctrl.gui)
        self.setAllowedAreas(Qt.LeftDockWidgetArea | Qt.RightDockWidgetArea)
        self.ctrl = ctrl
        self.main = QWidget()
        # complete status
        # diff (collapsable)
        # commit|merge|update|log
        # local path
        # remote path
        # push
        self.lo = QVBoxLayout(self.main)
        self.setWidget(self.main)


class MainGui(QMainWindow):
    def __init__(self, ctrl):
        QMainWindow.__init__(self)
        self.ctrl = ctrl
        self.tools = []
        self.currentTool = None
        self.central = QWidget()
        self.lo = QStackedLayout(self.central)
        self.setCentralWidget(self.central)
        # DOCKS
        self.pyhelper = PythonHelper("PyHelp", self, ctrl)
        self.pyhelper.hide()
        self.fileList = OpenFileList("Open Files", self, ctrl)
        self.fileList.hide()
        self.searchBox = SearchBox(self, ctrl)
        self.searchBox.hide()
        self.replaceBox = ReplaceBox(self, ctrl)
        self.replaceBox.hide()
        self.project = None
        self.projectStatus = ProjectStatus(ctrl)
        # MENUS
        # TODO: revert file
        # TODO: toggle more prevalent keyboard hints (overlay?)
        # TODO: About page with attributions (icon design, Qt, etc)
        # TODO: Second window
        # TODO: virtual desktops
        # TODO: when closing, list all files w/ changes
        # TODO: bookmarks dock
        # TODO: format paragraph
        # TODO: spell check
        # TODO: pylint/flake8
        # TODO: Jedi integration: syntax check, autocomplete, goto def, refactoring, search
        # TODO: init VCS: dock with diff, commit, push
        # TODO: CI: make.py, makefile, tests, dock widget with targets
        # TODO: projects remember dock states
        # TODO: add cells, web
        # TODO: factor out keybinding to config file
        # TODO: split into modules
        # TODO: rename
        # TODO: interpreter
        # TODO: Kanban todo tracking
        # TODO: built in distributed bug tracking
        self.actions = (
            (
                "&File",
                (
                    ("&New", self.new, QKeySequence("Ctrl+N")),
                    ("&Open", self.open, QKeySequence("Ctrl+O")),
                    ("&Save", self.save, QKeySequence("Ctrl+S")),
                    ("Save &As", self.saveAs, QKeySequence("Shift+Ctrl+S")),
                    ("&Close", self.closeTool, QKeySequence("Ctrl+C")),
                    ("-", None, None),
                    ("Create Pro&ject", self.createProject, None),
                    ("Open &Project", self.openProject, None),
                    ("-", None, None),
                    ("&Quit", self.quit, QKeySequence("Ctrl+Q")),
                ),
            ),
            (
                "&Edit",
                (
                    ("&Copy", self.copy, QKeySequence("Ctrl+C")),
                    ("Cu&t", self.cut, QKeySequence("Ctrl+X")),
                    ("&Paste", self.paste, QKeySequence("Ctrl+V")),
                    ("&Find", self.search, QKeySequence("Ctrl+F")),
                    ("&Replace", self.replace, QKeySequence("Ctrl+H")),
                ),
            ),
            (
                "&View",
                (
                    (
                        "Focus Current Tool",
                        lambda: self.currentTool.setFocus(),
                        QKeySequence("Escape"),
                    ),
                    ("Show Py&Help", self.togglePyHelp, QKeySequence("F1")),
                    ("Show Open &Files", self.toggleOpenFileList, QKeySequence("F2")),
                    ("Show &Project", self.toggleProject, QKeySequence("F3")),
                    ("-", None, None),
                    ("&Next Tool", self.nextTool, QKeySequence("Ctrl+PgDown")),
                    (
                        "&Back to Previous Tool",
                        self.backTool,
                        QKeySequence("Ctrl+PgUp"),
                    ),
                ),
            ),
        )
        self.actionBar = ActionBar(self.actions)
        self.setMenuBar(self.actionBar)
        # REMAINING SETUP
        self.displayUserMessage("Ready")
        # TODO: add hot reloading
        self.setStyleSheet(STYLESHEET)
        self.setWindowIcon(QIcon(RESOURCE("workbench.png")))
        if os.name == "nt":
            import ctypes

            myappid = "MyOrg.Mygui.1.0.0"  # Arbitrary
            ctypes.windll.shell32.SetCurrentProcessExplicitAppUserModelID(myappid)

    def spawnEditor(self):
        newTool = Editor(self.ctrl)
        self.tools.append(newTool)
        self.lo.addWidget(newTool)
        self.switchTool(newTool)

    def new(self):
        if self.currentTool.filename or self.currentTool.isUnsaved():
            self.spawnEditor()
            self.fileList.addItem(self.currentTool)

    def open(self, filename=None):
        if not filename:
            result = QFileDialog.getOpenFileName(
                self, options=QFileDialog.DontUseNativeDialog
            )
            if not result:
                return
            filename = result[0]
        # TODO: validate filename
        if (
            not self.currentTool
            or self.currentTool.filename
            or self.currentTool.isUnsaved()
        ):
            self.spawnEditor()
        else:
            self.fileList.removeItem(self.currentTool)
        self.currentTool.load(filename)
        self.fileList.addItem(self.currentTool)

    def save(self):
        if not self.currentTool.filename:
            self.saveAs()
        else:
            self.currentTool.save()

    def saveAs(self):
        result = QFileDialog.getSaveFileName(
            self, options=QFileDialog.DontUseNativeDialog
        )
        if not result:
            return
        self.fileList.removeItem(self.currentTool)
        self.currentTool.saveAs(result[0])
        self.fileList.addItem(self.currentTool)

    def closeTool(self):
        # TODO: check for unsaved changes
        toClose = self.currentTool
        index = self.tools.index(toClose)
        if index > 0:
            index -= 1
        self.tools.remove(toClose)
        if len(self.tools) == 0:
            self.spawnEditor()
        self.switchTool(self.tools[index])
        self.fileList.removeItem(toClose)
        self.lo.removeWidget(toClose)
        toClose.close()

    def createProject(self):
        result = QInputDialog().getText(
            self, "Enter a project result", "name:", QLineEdit.Normal, ""
        )
        if not result:
            return
        name = result[0]
        result = QFileDialog.getExistingDirectory(
            self,
            caption="Select the project directory",
            options=QFileDialog.DontUseNativeDialog | QFileDialog.ShowDirsOnly,
        )
        if not result:
            return
        path = result
        if not os.path.exists(path):
            return
        projectfile = os.path.join(path, ".project.yaml")
        if os.path.exists(projectfile):
            result = QMessageBox.question(
                self,
                "",
                "Project file already exists in {}. Do you want to overwrite?".format(
                    projectfile
                ),
            )
            if result != QMessageBox.Yes:
                return
        if os.path.exists(os.path.join(path, ".hg")):
            vcs = "hg"
        elif os.path.exists(os.path.join(path, ".git")):
            vcs = "git"
        else:
            vcs = ""
        self.project = Project(name=name, path=path, vcs=vcs, savedStates=[])
        open(projectfile, "w").write(yaml.dump(asdict(self.project)))
        self.projectStatus.setProject(self.project)

    def openProject(self, directory=None):
        # TODO: filter
        if not directory:
            result = QFileDialog.getExistingDirectory(
                self, options=QFileDialog.DontUseNativeDialog
            )
            if not result:
                return
            directory = result
        fn = os.path.join(directory, ".project.yaml")
        if not os.path.exists(fn):
            return
        self.project = Project(**yaml.safe_load(open(fn).read()))
        self.project.path = os.path.dirname(fn)
        self.projectStatus.setProject(self.project)
        for record in self.project.savedStates:
            self.open(record["filename"])
            self.currentTool.loadState(record)

    def quit(self):
        # TODO: check for unsaved files
        if self.project:
            toolStates = []
            for tool in self.tools:
                # TODO: Need special handling here if .project.yaml is the open file
                toolStates.append(tool.saveState())
            self.project.savedStates = toolStates
            with open(os.path.join(self.project.path, ".project.yaml"), "w") as outfile:
                outfile.write(yaml.dump(asdict(self.project)))
        self.close()

    def copy(self):
        self.currentTool.copy()

    def cut(self):
        self.currentTool.cut()

    def paste(self):
        self.currentTool.paste()

    def search(self):
        self.toggleDock(self.searchBox, Qt.BottomDockWidgetArea)

    def replace(self):
        self.toggleDock(self.replaceBox, Qt.BottomDockWidgetArea)

    def displayUserMessage(self, message):
        self.statusBar().showMessage(message, 5000)

    def updateTitleBar(self, message):
        self.setWindowTitle(message)

    def toggleDock(self, dock, area=Qt.RightDockWidgetArea):
        # TODO: clearer indication of focus
        if self.dockWidgetArea(dock) == Qt.NoDockWidgetArea:
            self.addDockWidget(area, dock)
            dock.show()
        elif dock.hasFocus():
            self.removeDockWidget(dock)
        else:
            dock.setFocus()

    def toggleOpenFileList(self):
        self.toggleDock(self.fileList)

    def togglePyHelp(self):
        self.toggleDock(self.pyhelper)

    def toggleProject(self):
        self.toggleDock(self.projectStatus)

    def switchTool(self, tool):
        self.currentTool = tool
        self.lo.setCurrentWidget(tool)
        self.currentTool.updateTitleBar()
        # TODO: update OpenFiles

    def nextTool(self):
        i = self.tools.index(self.currentTool)
        i += 1
        if i >= len(self.tools):
            i = 0
        self.switchTool(self.tools[i])

    def backTool(self):
        i = self.tools.index(self.currentTool)
        i -= 1
        if i < 0:
            i = len(self.tools) - 1
        self.switchTool(self.tools[i])


class Main:
    def __init__(self):
        parser = argparse.ArgumentParser(
            prog="shokunin", description="A craftsman's development environment."
        )
        parser.add_argument("-p", "--project", type=str, help="project to load")
        parser.add_argument("files", nargs="*", type=str, help="files to load")
        self.args = parser.parse_args()
        self.gui = None
        app = QApplication([])
        self.gui = MainGui(self)
        self.gui.showMaximized()
        self.gui.spawnEditor()

        if self.args.project:
            self.gui.openProject(self.args.project)
        if self.args.files:
            for f in self.args.files:
                self.gui.open(f)

        app.exec_()


if __name__ == "__main__":
    Main()
