import re
import os
import sys
import shlex
import yaml
import argparse
import subprocess
from collections import defaultdict

from PySide2.QtWidgets import (
    QApplication,
    QWidget,
    QMainWindow,
    QTextEdit,
    QLineEdit,
    QStatusBar,
    QMenu,
    QMenuBar,
    QToolTip,
    QVBoxLayout,
    QHBoxLayout,
    QLabel,
    QAction,
)

from PySide2.QtGui import (
    QSyntaxHighlighter,
    QTextDocument,
    QTextCharFormat,
    QBrush,
    QColor,
    QTextBlockUserData,
    QTextCursor,
    QKeySequence,
)
from PySide2.QtCore import Qt, QRegularExpression

from pygments.token import Token
from pygments.lexers import PythonLexer

""" More features:
    TODO: Todo list scraped from comments
    TODO: git/hg stat/diff pane'
    TODO: python console (REPL)
    TODO: python output console
    TODO: action bar changes when you move focus
    TODO: action bar changes color
    TODO: add modeline
    TODO: project file listing
    TODO: create project yaml files
    TODO: split up class into files
    TODO: open file from project file list
    TODO: add/remove files from project
    TODO: add/remove files from VCS
    TODO: in app commit
    TODO: run tests from app
    TODO: make search more like FF quick search
        find next/back ([Shift]+F3)
        goto (Return)
        changing focus makes it go away (Esc)
    TODO: get rid of resize handle
    TODO: Line numbers
    TODO: virtual desktops
    TODO: i3 style layouts
    TODO: add WM layouts to [project].yaml files
"""


class MagicMenuBar:
    """ Wait, isn't magic a bad thing? 
        When a Tool (including the CLI) gets the focus it calls MagicMenuBar.rebuild
        passing itself. rebuild references Settings.bindings which contains all the
        possible actions of every Tool, including top level actions associated with
        Main/MainGui. The tool provides a list of all actions that it implements, by
        action key, with handler functions. MagicMenuBar then builds the menu with
        language, shortcuts, and order from bindings.
    """

    def __init__(self, ctrl):
        self.ctrl = ctrl

    def rebuild(self, focusTool):
        bar = QMenuBar()
        actions = defaultdict(list)
        bindings = self.ctrl.settings.bindings

        # tools whose actions are always on the menu
        main = self.ctrl.gui
        cli = self.ctrl.gui.statusbar.cli
        if focusTool == cli:
            tools_consider = (main, cli)
        else:
            tools_consider = (main, cli, focusTool)

        for tool in tools_consider:
            print(tool.toolKey, tool.mode)
            for actionKey, actionConf in bindings[tool.toolKey][tool.mode].items():
                if not actionKey in tool.actions:
                    raise Exception("Unknown action: {}".format(actionKey))
                handler = tool.actions[actionKey]
                cluster, text, shortcut = actionConf
                actions[cluster].append((text, QKeySequence(shortcut), handler))

        for menuText, clusters in self.ctrl.settings.menus.items():
            empty = True
            menu = QMenu(menuText)
            for cluster in clusters:
                for text, shortcut, handler in actions[cluster]:
                    menu.addAction(text, handler, shortcut)
                    empty = False
            if not empty:
                bar.addMenu(menu)
        # top level actions
        print(actions)
        for text, shortcut, handler in actions["top"]:
            action = QAction(text)
            action.triggered.connect(handler)
            action.setShortcut(shortcut)
            bar.addAction(text, handler)

        self.ctrl.gui.setMenuBar(bar)

    def getMenu(self, key):
        if key == "top":
            return self
        return self.menus[key]

    def addMenu(self, key, title):
        menu = QMenu(title)
        self.menus[key] = menu
        QMenuBar.addMenu(self, menu)
        return menu


def makeFormat(foreground, background=None, italic=False, bold=False, underline=False):
    """ create a QTextCharFormat based on some common attributes """
    tcFormat = QTextCharFormat()
    tcFormat.setForeground(QBrush(QColor(foreground)))
    if background:
        tcFormat.setBackground(QBrush(QColor(background)))
    if italic:
        tcFormat.setFontItalic(True)
    if bold:
        tcFormat.setFontWeight(100)
    return tcFormat


class UserData(QTextBlockUserData):
    """ Stores the highlighter state of the current textblock """

    def __init__(self, stack):
        QTextBlockUserData.__init__(self)
        self.stack = stack


class Highlighter(QSyntaxHighlighter):
    """ Pygments based highlighter class """

    def __init__(self, doc):
        QSyntaxHighlighter.__init__(self, doc)
        self.lex = PythonLexer()
        # WARNING: a typo on Token.XX not throw an error
        self.styles = {
            Token.Comment: makeFormat(0x00AA00, italic=True),
            Token.Literal.String.Doc: makeFormat(0x00AA00, italic=True),
            Token.Keyword: makeFormat(0x6600AA),
            Token.Name: makeFormat(0x00),
            Token.Name.Function: makeFormat(0xAA5500),
            Token.Name.Class: makeFormat(0xAA5500, bold=True),
            Token.Literal.String: makeFormat(0x1100AA),
            Token.Literal: makeFormat(0xAA0000),
            Token.Operator: makeFormat(0xAA0000),
        }

    def styleMatcher(self, token):
        match = None
        for tokenType in self.styles:
            if token in tokenType and (not match or len(tokenType) > len(match)):
                match = tokenType
        return match

    def highlightBlock(self, text):
        block = self.currentBlock()
        userdata = block.previous().userData()
        if not userdata or not userdata.stack:
            stack = ("root",)
        else:
            stack = userdata.stack
        for r in self.lex.get_tokens_unprocessed(text, stack):
            i, tt, v, stack = r
            match = self.styleMatcher(tt)
            if match:
                self.setFormat(i, len(v), self.styles[match])
        self.setCurrentBlockUserData(UserData(stack))


class TitleBar(QWidget):
    def __init__(self, ctrl, attached, fields):
        """ TitleBar provides a modeline for the tool
            attached = the tool attached to this titlebar
            fields = list of (strings) names of fields
        """
        QWidget.__init__(self)
        self.ctrl = ctrl
        self.attached = attached
        self.attached.titlebar = self
        self.lo = QHBoxLayout(self)
        self.lo.setSpacing(0)
        self.lo.setContentsMargins(0, 0, 0, 0)
        self.fields = {}
        for fieldKey in fields:
            self.fields[fieldKey] = QLabel()
            self.lo.addWidget(self.fields[fieldKey])

    def setFieldText(self, fieldKey, value):
        self.fields[fieldKey].setText(value)


class EditorWidget(QWidget):
    def __init__(self, ctrl, editor):
        QWidget.__init__(self)
        self.lo = QVBoxLayout(self)
        self.lo.setSpacing(0)
        self.lo.setContentsMargins(0, 0, 0, 0)
        self.titlebar = TitleBar(ctrl, self, ("filename", "mode"))
        self.lo.addWidget(self.titlebar)
        self.edit = editor
        self.lo.addWidget(self.edit)
        # TODO: add line numbers here


class Editor(QTextEdit):
    def __init__(self, ctrl):
        QTextEdit.__init__(self)
        self.ctrl = ctrl
        self.toolKey = "edit"
        self.mode = "insert"
        self.filename = None

        self.actions = {
            "save": self.save,
            "open": self.open,
            "new": self.new,
        }

        self.wrapperWidget = EditorWidget(ctrl, self)
        self.titlebar = self.wrapperWidget.titlebar

        # TODO: add these to settings
        self.autoIndentRE = ".+:$"
        self.tab = "    "
        self.newline = "\n"
        # TODO: add an autosave timer
        # TODO: bookmarks
        # TODO: goto
        # TODO: make cursor visible when out of focus?

    def focusInEvent(self, event):
        self.ctrl.gui.actionBar.rebuild(self)

    def open(self, filename):
        self.setPlainText(open(filename).read())
        self.filename = filename
        self.titlebar.setFieldText("filename", filename)
        if self.filename.endswith(".py"):
            self.hl = Highlighter(self.document())

    def save(self):
        if self.filename:
            open(self.filename, "w").write(self.toPlainText())
            subprocess.run(shlex.split("python3 -m black " + self.filename))
            pos = self.textCursor().position()
            scrollPos = self.verticalScrollBar().value()
            self.open(self.filename)
            tc = self.textCursor()
            tc.setPosition(pos)
            self.setTextCursor(tc)
            self.verticalScrollBar().setValue(scrollPos)

    def new(self):
        self.filename = None
        self.setPlainText("")

    def search(self, exp, direction=None):
        if direction is None:
            tc = self.textCursor()
            tc.setPosition(tc.anchor())
            # if you don't do this it searches from the end of the current selection
            # messing up search as you type
            self.setTextCursor(tc)
        if direction == "back":
            self.find(QRegularExpression(exp), QTextDocument.FindBackward)
        else:
            self.find(QRegularExpression(exp))

    def keyPressEvent(self, event):
        key = event.key()
        # TODO: fix autotab: inserting a new line in the middle of a line is broken
        # TODO: make it an action instead
        if key == Qt.Key_Tab:
            self.insertPlainText(self.tab)
        elif key == Qt.Key_Backtab:
            tc = self.textCursor()
            for f in range(len(self.tab)):
                tc.movePosition(QTextCursor.PreviousCharacter, QTextCursor.KeepAnchor)
                possibleTab = tc.selectedText()
                if possibleTab == self.tab:
                    tc.deleteChar()
        elif key == Qt.Key_Return:
            tc = self.textCursor()
            tc.movePosition(QTextCursor.StartOfLine, QTextCursor.KeepAnchor)
            line = tc.selectedText()
            indent = 0
            while line.startswith(self.tab):
                indent += 1
                line = line[len(self.tab) :]
            if re.match(self.autoIndentRE, line):
                indent += 1
            tc.movePosition(QTextCursor.EndOfLine)
            tc.insertText(self.newline + self.tab * indent)
        else:
            QTextEdit.keyPressEvent(self, event)


class Cli(QLineEdit):
    def __init__(self, ctrl):
        QLineEdit.__init__(self)
        self.ctrl = ctrl
        self.toolKey = "cli"
        self.mode = "normal"
        self.deactivate()

        self.actions = {
            "search": lambda: self.activate("search"),
            "leave search": self.deactivate,
            "forward match": lambda: self.updateSearch("forward"),
        }

        self.textChanged.connect(self.updateSearch)
        self.prompt = ""

    def focusInEvent(self, event):
        self.ctrl.gui.actionBar.rebuild(self)

    def focusOutEvent(self, event):
        self.deactivate()

    def activate(self, promptKey):
        self.prompt = self.ctrl.settings.config["prompts"][promptKey]
        if promptKey == "search":
            # enable search as you type
            self.mode = "search"
        else:
            self.mode = "normal"
        self.setText(self.prompt)
        self.setReadOnly(False)
        self.setFocus()

    def deactivate(self):
        self.mode = "normal"
        self.setText("")
        self.setReadOnly(True)
        self.clearFocus()

    def updateSearch(self, direction=None):
        # if in search mode update search as you type
        if self.mode == "search":
            # update the search as you type string
            needle = self.text()[len(self.prompt) :]
            if direction in ("forward", "back"):
                self.ctrl.searchInTool(needle, direction)
            else:
                self.ctrl.searchInTool(needle)

    def keyPressEvent(self, event):
        key = event.key()
        pos = self.cursorPosition()
        # block movement that will move the cursor past the prompt
        if key == Qt.Key_Home:
            self.setCursorPosition(len(self.prompt))
            return False
        if pos == len(self.prompt):
            if key in [Qt.Key_Backspace, Qt.Key_Left]:
                return False
        # TODO: make this an action
        if key == Qt.Key_Return:
            self.updateSearch("forward")
        return QLineEdit.keyPressEvent(self, event)


class SuperStatusBar(QStatusBar):
    def __init__(self, ctrl):
        QStatusBar.__init__(self)
        self.ctrl = ctrl
        self.cli = Cli(ctrl)
        self.addWidget(self.cli)


STYLESHEET = """
QWidget {{
    background-color: #{background:06x};
    color: #{foreground:06x};
    font-family: "{font}";
    font-size: {font-size}px;
    border: 1px solid {border};
    padding: {padding};
    margin: {margins};
}}
"""


class MainGui(QMainWindow):
    def __init__(self, ctrl):
        QMainWindow.__init__(self)
        self.ctrl = ctrl
        self.toolKey = "main"
        self.mode = "normal"
        self.actions = {
            "quit": self.ctrl.quit,
        }

        self.style(ctrl.settings.style)

        self.statusbar = SuperStatusBar(ctrl)
        self.setStatusBar(self.statusbar)

        self.actionBar = MagicMenuBar(ctrl)

    def style(self, yamlstyle):
        ss = STYLESHEET.format(**yamlstyle)
        self.setStyleSheet(ss)

    def spawnTool(self, cls):
        tool = cls(self.ctrl)
        # TODO: implement window management
        if hasattr(tool, "wrapperWidget"):
            self.setCentralWidget(tool.wrapperWidget)
        else:
            self.setCentralWidget(tool)
        return tool


class Settings:
    def __init__(self, path):
        self.config = self.loadYaml(path, "config.yaml")
        self.styles = self.loadYaml(path, "style.yaml")
        self.style = self.styles[self.config["style"]]
        self.bindings = self.loadYaml(path, "bindings.yaml")
        self.menus = self.loadYaml(path, "menus.yaml")

    def loadYaml(self, path, filename):
        return yaml.load(open(os.path.join(path, filename)).read())


class Main:
    def __init__(self):
        self.tools = {}
        self.hasFocus = None
        self.gui = None

        cliparser = argparse.ArgumentParser(
            description="ed2 - My second attempt at a Qt based code editor"
        )
        cliparser.add_argument(
            "filename", nargs="?", help="The file to load", default=None
        )
        arguments = cliparser.parse_args()
        path = os.path.join(os.path.split(os.path.abspath(__file__))[0], "data")
        self.settings = Settings(path)
        app = QApplication([])
        self.gui = MainGui(self)
        self.gui.show()

        self.openEditor(arguments.filename)

        app.exec_()

    def quit(self):
        # Check open tools for unsaved work
        # or running processes
        # if found, prompt the user
        # o/w
        self.gui.close()

    def openEditor(self, filename=None):
        tool = self.gui.spawnTool(Editor)
        if filename:
            tool.open(filename)
        self.tools[id(tool)] = tool
        self.hasFocus = tool
        tool.setFocus()

    def searchInTool(self, exp, direction=None):
        """ direction may be 
            None = Search as you type
            Next = forward in document
            Back = backwards in document
        """

        if self.hasFocus:
            self.hasFocus.search(exp, direction)


if __name__ == "__main__":
    Main()
